(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.Core = {}));
})(this, (function (exports) { 'use strict';

    var MASKITO_DEFAULT_ELEMENT_PREDICATE = function (e) {
      return e.querySelector('input,textarea') || e;
    };

    var MASKITO_DEFAULT_OPTIONS = {
      mask: /^.*$/,
      preprocessors: [],
      postprocessors: [],
      plugins: [],
      overwriteMode: 'shift'
    };

    /******************************************************************************
    Copyright (c) Microsoft Corporation.

    Permission to use, copy, modify, and/or distribute this software for any
    purpose with or without fee is hereby granted.

    THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
    REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
    AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
    INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
    LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
    OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
    PERFORMANCE OF THIS SOFTWARE.
    ***************************************************************************** */
    /* global Reflect, Promise, SuppressedError, Symbol */

    var extendStatics = function(d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };

    function __extends(d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    }

    var __assign = function() {
        __assign = Object.assign || function __assign(t) {
            for (var s, i = 1, n = arguments.length; i < n; i++) {
                s = arguments[i];
                for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p)) t[p] = s[p];
            }
            return t;
        };
        return __assign.apply(this, arguments);
    };

    function __read(o, n) {
        var m = typeof Symbol === "function" && o[Symbol.iterator];
        if (!m) return o;
        var i = m.call(o), r, ar = [], e;
        try {
            while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
        }
        catch (error) { e = { error: error }; }
        finally {
            try {
                if (r && !r.done && (m = i["return"])) m.call(i);
            }
            finally { if (e) throw e.error; }
        }
        return ar;
    }

    function __spreadArray(to, from, pack) {
        if (pack || arguments.length === 2) for (var i = 0, l = from.length, ar; i < l; i++) {
            if (ar || !(i in from)) {
                if (!ar) ar = Array.prototype.slice.call(from, 0, i);
                ar[i] = from[i];
            }
        }
        return to.concat(ar || Array.prototype.slice.call(from));
    }

    typeof SuppressedError === "function" ? SuppressedError : function (error, suppressed, message) {
        var e = new Error(message);
        return e.name = "SuppressedError", e.error = error, e.suppressed = suppressed, e;
    };

    var MaskHistory =
    /** @class */
    function () {
      function MaskHistory() {
        this.now = null;
        this.past = [];
        this.future = [];
      }

      MaskHistory.prototype.undo = function () {
        var state = this.past.pop();

        if (state && this.now) {
          this.future.push(this.now);
          this.updateElement(state, 'historyUndo');
        }
      };

      MaskHistory.prototype.redo = function () {
        var state = this.future.pop();

        if (state && this.now) {
          this.past.push(this.now);
          this.updateElement(state, 'historyRedo');
        }
      };

      MaskHistory.prototype.updateHistory = function (state) {
        if (!this.now) {
          this.now = state;
          return;
        }

        var isValueChanged = this.now.value !== state.value;
        var isSelectionChanged = this.now.selection.some(function (item, index) {
          return item !== state.selection[index];
        });

        if (!isValueChanged && !isSelectionChanged) {
          return;
        }

        if (isValueChanged) {
          this.past.push(this.now);
          this.future = [];
        }

        this.now = state;
      };

      MaskHistory.prototype.updateElement = function (state, inputType) {
        this.now = state;
        this.updateElementState(state, {
          inputType: inputType,
          data: null
        });
      };

      return MaskHistory;
    }();

    function areElementValuesEqual(sampleState) {
      var states = [];

      for (var _i = 1; _i < arguments.length; _i++) {
        states[_i - 1] = arguments[_i];
      }

      return states.every(function (_a) {
        var value = _a.value;
        return value === sampleState.value;
      });
    }
    function areElementStatesEqual(sampleState) {
      var states = [];

      for (var _i = 1; _i < arguments.length; _i++) {
        states[_i - 1] = arguments[_i];
      }

      return states.every(function (_a) {
        var value = _a.value,
            selection = _a.selection;
        return value === sampleState.value && selection[0] === sampleState.selection[0] && selection[1] === sampleState.selection[1];
      });
    }

    function applyOverwriteMode(_a, newCharacters, mode) {
      var value = _a.value,
          selection = _a.selection;

      var _b = __read(selection, 2),
          from = _b[0],
          to = _b[1];

      var computedMode = typeof mode === 'function' ? mode({
        value: value,
        selection: selection
      }) : mode;
      return {
        value: value,
        selection: computedMode === 'replace' ? [from, from + newCharacters.length] : [from, to]
      };
    }

    function isFixedCharacter(char) {
      return typeof char === 'string';
    }

    function getLeadingFixedCharacters(mask, validatedValuePart, newCharacter, initialElementState) {
      var leadingFixedCharacters = '';

      for (var i = validatedValuePart.length; i < mask.length; i++) {
        var charConstraint = mask[i];
        var isInitiallyExisted = (initialElementState === null || initialElementState === void 0 ? void 0 : initialElementState.value[i]) === charConstraint;

        if (!isFixedCharacter(charConstraint) || charConstraint === newCharacter && !isInitiallyExisted) {
          return leadingFixedCharacters;
        }

        leadingFixedCharacters += charConstraint;
      }

      return leadingFixedCharacters;
    }

    function validateValueWithMask(value, maskExpression) {
      if (Array.isArray(maskExpression)) {
        return value.length === maskExpression.length && Array.from(value).every(function (char, i) {
          var charConstraint = maskExpression[i];
          return isFixedCharacter(charConstraint) ? char === charConstraint : char.match(charConstraint);
        });
      }

      return maskExpression.test(value);
    }

    function guessValidValueByPattern(elementState, mask, initialElementState) {
      var maskedFrom = null;
      var maskedTo = null;
      var maskedValue = Array.from(elementState.value).reduce(function (validatedCharacters, char, charIndex) {
        var leadingCharacters = getLeadingFixedCharacters(mask, validatedCharacters, char, initialElementState);
        var newValidatedChars = validatedCharacters + leadingCharacters;
        var charConstraint = mask[newValidatedChars.length];

        if (isFixedCharacter(charConstraint)) {
          return newValidatedChars + charConstraint;
        }

        if (!char.match(charConstraint)) {
          return newValidatedChars;
        }

        if (maskedFrom === null && charIndex >= elementState.selection[0]) {
          maskedFrom = newValidatedChars.length;
        }

        if (maskedTo === null && charIndex >= elementState.selection[1]) {
          maskedTo = newValidatedChars.length;
        }

        return newValidatedChars + char;
      }, '');
      var trailingFixedCharacters = getLeadingFixedCharacters(mask, maskedValue, '', initialElementState);
      return {
        value: validateValueWithMask(maskedValue + trailingFixedCharacters, mask) ? maskedValue + trailingFixedCharacters : maskedValue,
        selection: [maskedFrom !== null && maskedFrom !== void 0 ? maskedFrom : maskedValue.length, maskedTo !== null && maskedTo !== void 0 ? maskedTo : maskedValue.length]
      };
    }

    function guessValidValueByRegExp(_a, maskRegExp) {
      var value = _a.value,
          selection = _a.selection;

      var _b = __read(selection, 2),
          from = _b[0],
          to = _b[1];

      var newFrom = from;
      var newTo = to;
      var validatedValue = Array.from(value).reduce(function (validatedValuePart, char, i) {
        var newPossibleValue = validatedValuePart + char;

        if (from === i) {
          newFrom = validatedValuePart.length;
        }

        if (to === i) {
          newTo = validatedValuePart.length;
        }

        return newPossibleValue.match(maskRegExp) ? newPossibleValue : validatedValuePart;
      }, '');
      return {
        value: validatedValue,
        selection: [newFrom, newTo]
      };
    }

    function calibrateValueByMask(elementState, mask, initialElementState) {
      if (initialElementState === void 0) {
        initialElementState = null;
      }

      if (validateValueWithMask(elementState.value, mask)) {
        return elementState;
      }

      var _a = Array.isArray(mask) ? guessValidValueByPattern(elementState, mask, initialElementState) : guessValidValueByRegExp(elementState, mask),
          value = _a.value,
          selection = _a.selection;

      return {
        selection: selection,
        value: Array.isArray(mask) ? value.slice(0, mask.length) : value
      };
    }

    function removeFixedMaskCharacters(initialElementState, mask) {
      if (!Array.isArray(mask)) {
        return initialElementState;
      }

      var _a = __read(initialElementState.selection, 2),
          from = _a[0],
          to = _a[1];

      var selection = [];
      var unmaskedValue = Array.from(initialElementState.value).reduce(function (rawValue, char, i) {
        var charConstraint = mask[i];

        if (i === from) {
          selection.push(rawValue.length);
        }

        if (i === to) {
          selection.push(rawValue.length);
        }

        return isFixedCharacter(charConstraint) && charConstraint === char ? rawValue : rawValue + char;
      }, '');

      if (selection.length < 2) {
        selection.push.apply(selection, __spreadArray([], __read(new Array(2 - selection.length).fill(unmaskedValue.length))));
      }

      return {
        value: unmaskedValue,
        selection: [selection[0], selection[1]]
      };
    }

    var MaskModel =
    /** @class */
    function () {
      function MaskModel(initialElementState, maskOptions) {
        this.initialElementState = initialElementState;
        this.maskOptions = maskOptions;
        this.value = '';
        this.selection = [0, 0];

        var _a = calibrateValueByMask(initialElementState, this.getMaskExpression(initialElementState)),
            value = _a.value,
            selection = _a.selection;

        this.value = value;
        this.selection = selection;
      }

      MaskModel.prototype.addCharacters = function (_a, newCharacters) {
        var _b = __read(_a, 2),
            from = _b[0],
            to = _b[1];

        var value = this.value;
        var maskExpression = this.getMaskExpression({
          value: value.slice(0, from) + newCharacters + value.slice(to),
          selection: [from + newCharacters.length, from + newCharacters.length]
        });
        var initialElementState = {
          value: value,
          selection: [from, to]
        };
        var unmaskedElementState = removeFixedMaskCharacters(initialElementState, maskExpression);

        var _c = __read(applyOverwriteMode(unmaskedElementState, newCharacters, this.maskOptions.overwriteMode).selection, 2),
            unmaskedFrom = _c[0],
            unmaskedTo = _c[1];

        var newUnmaskedLeadingValuePart = unmaskedElementState.value.slice(0, unmaskedFrom) + newCharacters;
        var newCaretIndex = newUnmaskedLeadingValuePart.length;
        var maskedElementState = calibrateValueByMask({
          value: newUnmaskedLeadingValuePart + unmaskedElementState.value.slice(unmaskedTo),
          selection: [newCaretIndex, newCaretIndex]
        }, maskExpression, initialElementState);
        var isInvalidCharsInsertion = // eslint-disable-next-line @typescript-eslint/prefer-string-starts-ends-with
        value.slice(0, unmaskedFrom) === calibrateValueByMask({
          value: newUnmaskedLeadingValuePart,
          selection: [newCaretIndex, newCaretIndex]
        }, maskExpression, initialElementState).value;

        if (isInvalidCharsInsertion || areElementStatesEqual(this, maskedElementState) // If typing new characters does not change value
        ) {
          throw new Error('Invalid mask value');
        }

        this.value = maskedElementState.value;
        this.selection = maskedElementState.selection;
      };

      MaskModel.prototype.deleteCharacters = function (_a) {
        var _b = __read(_a, 2),
            from = _b[0],
            to = _b[1];

        if (from === to || !to) {
          return;
        }

        var value = this.value;
        var maskExpression = this.getMaskExpression({
          value: value.slice(0, from) + value.slice(to),
          selection: [from, from]
        });
        var initialElementState = {
          value: value,
          selection: [from, to]
        };
        var unmaskedElementState = removeFixedMaskCharacters(initialElementState, maskExpression);

        var _c = __read(unmaskedElementState.selection, 2),
            unmaskedFrom = _c[0],
            unmaskedTo = _c[1];

        var newUnmaskedValue = unmaskedElementState.value.slice(0, unmaskedFrom) + unmaskedElementState.value.slice(unmaskedTo);
        var maskedElementState = calibrateValueByMask({
          value: newUnmaskedValue,
          selection: [unmaskedFrom, unmaskedFrom]
        }, maskExpression, initialElementState);
        this.value = maskedElementState.value;
        this.selection = maskedElementState.selection;
      };

      MaskModel.prototype.getMaskExpression = function (elementState) {
        var mask = this.maskOptions.mask;
        return typeof mask === 'function' ? mask(elementState) : mask;
      };

      return MaskModel;
    }();

    var EventListener =
    /** @class */
    function () {
      function EventListener(element) {
        this.element = element;
        this.listeners = [];
      }

      EventListener.prototype.listen = function (eventType, fn, options) {
        var _this = this;

        var untypedFn = fn;
        this.element.addEventListener(eventType, untypedFn, options);
        this.listeners.push(function () {
          return _this.element.removeEventListener(eventType, untypedFn);
        });
      };

      EventListener.prototype.destroy = function () {
        this.listeners.forEach(function (stopListen) {
          return stopListen();
        });
      };

      return EventListener;
    }();

    /**
     * Checks if the passed keyboard event match the required hotkey.
     *
     * We intentionally use legacy {@link KeyboardEvent#keyCode `keyCode`} property. It is more
     * "keyboard-layout"-independent than {@link KeyboardEvent#key `key`} or {@link KeyboardEvent#code `code`} properties.
     *
     * @example
     * input.addEventListener('keydown', (event) => {
     *     if (isHotkey(event, HotkeyModifier.CTRL | HotkeyModifier.SHIFT, HotkeyCode.Z)) {
     *         // redo hotkey pressed
     *     }
     * })
     *
     * @see {@link https://github.com/taiga-family/maskito/issues/315 `KeyboardEvent#code` issue}
     *
     * @return will return `true` only if the {@link HotkeyCode} matches and only the necessary
     * {@link HotkeyModifier modifiers} have been pressed
     */
    function isHotkey(event, modifiers, hotkeyCode) {
      return event.ctrlKey === !!(modifiers & 1
      /* CTRL */
      ) && event.altKey === !!(modifiers & 2
      /* ALT */
      ) && event.shiftKey === !!(modifiers & 4
      /* SHIFT */
      ) && event.metaKey === !!(modifiers & 8
      /* META */
      ) && event.keyCode === hotkeyCode;
    }

    function isRedo(event) {
      return isHotkey(event, 1
      /* CTRL */
      , 89
      /* Y */
      ) || // Windows
      isHotkey(event, 1
      /* CTRL */
      | 4
      /* SHIFT */
      , 90
      /* Z */
      ) || // Windows & Android
      isHotkey(event, 8
      /* META */
      | 4
      /* SHIFT */
      , 90
      /* Z */
      ) // macOS & iOS
      ;
    }
    function isUndo(event) {
      return isHotkey(event, 1
      /* CTRL */
      , 90
      /* Z */
      ) || // Windows & Android
      isHotkey(event, 8
      /* META */
      , 90
      /* Z */
      ) // macOS & iOS
      ;
    }

    /**
     * "beforeinput" is more appropriate event for preprocessing of the input masking (than `keydown`):
     * - `keydown` is not triggered by predictive text from native mobile keyboards.
     * - `keydown` is triggered by system key combinations (we don't need them, and they should be manually filtered).
     * - Dropping text inside input triggers `beforeinput` (but not `keydown`).
     * ___
     * "beforeinput" is not supported by Chrome 49+ (only from 60+) and by Firefox 52+ (only from 87+).
     *
     * @see https://caniuse.com/?search=beforeinput
     * @see https://taiga-ui.dev/browser-support
     */
    function isBeforeInputEventSupported(element) {
      return 'onbeforeinput' in element;
    }

    function isEventProducingCharacter(_a) {
      var key = _a.key,
          ctrlKey = _a.ctrlKey,
          metaKey = _a.metaKey,
          altKey = _a.altKey;
      var isSystemKeyCombinations = ctrlKey || metaKey || altKey;
      var isSingleUnicodeChar = /^.$/u.test(key); // 4-byte characters case (e.g. smile)

      return !isSystemKeyCombinations && key !== 'Backspace' && isSingleUnicodeChar;
    }

    function getLineSelection(_a, isForward) {
      var value = _a.value,
          selection = _a.selection;

      var _b = __read(selection, 2),
          from = _b[0],
          to = _b[1];

      if (from !== to) {
        return [from, to];
      }

      var nearestBreak = isForward ? value.slice(from).indexOf('\n') + 1 || value.length : value.slice(0, to).lastIndexOf('\n') + 1;
      var selectFrom = isForward ? from : nearestBreak;
      var selectTo = isForward ? nearestBreak : to;
      return [selectFrom, selectTo];
    }

    function getNotEmptySelection(_a, isForward) {
      var value = _a.value,
          selection = _a.selection;

      var _b = __read(selection, 2),
          from = _b[0],
          to = _b[1];

      if (from !== to) {
        return [from, to];
      }

      var notEmptySelection = isForward ? [from, to + 1] : [from - 1, to];
      return notEmptySelection.map(function (x) {
        return Math.min(Math.max(x, 0), value.length);
      });
    }

    var TRAILING_SPACES_REG = /\s+$/g;
    var LEADING_SPACES_REG = /^\s+/g;
    var SPACE_REG = /\s/;
    function getWordSelection(_a, isForward) {
      var value = _a.value,
          selection = _a.selection;

      var _b = __read(selection, 2),
          from = _b[0],
          to = _b[1];

      if (from !== to) {
        return [from, to];
      }

      if (isForward) {
        var valueAfterSelectionStart = value.slice(from);

        var _c = __read(valueAfterSelectionStart.match(LEADING_SPACES_REG) || [''], 1),
            leadingSpaces = _c[0];

        var nearestWordEndIndex = valueAfterSelectionStart.replace(LEADING_SPACES_REG, '') // TODO replace with `String.trimStart` after bumping Firefox to 61+
        .search(SPACE_REG);
        return [from, nearestWordEndIndex !== -1 ? from + leadingSpaces.length + nearestWordEndIndex : value.length];
      }

      var valueBeforeSelectionEnd = value.slice(0, to);

      var _d = __read(valueBeforeSelectionEnd.match(TRAILING_SPACES_REG) || [''], 1),
          trailingSpaces = _d[0];

      var selectedWordLength = valueBeforeSelectionEnd.replace(TRAILING_SPACES_REG, '') // TODO replace with `String.trimEnd` after bumping Firefox to 61+
      .split('').reverse().findIndex(function (char) {
        return char.match(SPACE_REG);
      });
      return [selectedWordLength !== -1 ? to - trailingSpaces.length - selectedWordLength : 0, to];
    }

    /* eslint-disable @typescript-eslint/ban-types */

    /**
     * @internal
     */

    function maskitoPipe(processors) {
      if (processors === void 0) {
        processors = [];
      }

      return function (initialData) {
        var readonlyArgs = [];

        for (var _i = 1; _i < arguments.length; _i++) {
          readonlyArgs[_i - 1] = arguments[_i];
        }

        return processors.reduce(function (data, fn) {
          return __assign(__assign({}, data), fn.apply(void 0, __spreadArray([data], __read(readonlyArgs))));
        }, initialData);
      };
    }

    function maskitoTransform(valueOrState, maskitoOptions) {
      var options = __assign(__assign({}, MASKITO_DEFAULT_OPTIONS), maskitoOptions);

      var preprocessor = maskitoPipe(options.preprocessors);
      var postprocessor = maskitoPipe(options.postprocessors);
      var initialElementState = typeof valueOrState === 'string' ? {
        value: valueOrState,
        selection: [0, 0]
      } : valueOrState;
      var elementState = preprocessor({
        elementState: initialElementState,
        data: ''
      }, 'validation').elementState;
      var maskModel = new MaskModel(elementState, options);

      var _a = postprocessor(maskModel, initialElementState),
          value = _a.value,
          selection = _a.selection;

      return typeof valueOrState === 'string' ? value : {
        value: value,
        selection: selection
      };
    }

    var Maskito =
    /** @class */
    function (_super) {
      __extends(Maskito, _super);

      function Maskito(element, maskitoOptions) {
        var _this = _super.call(this) || this;

        _this.element = element;
        _this.maskitoOptions = maskitoOptions;
        _this.isTextArea = _this.element.nodeName === 'TEXTAREA';
        _this.eventListener = new EventListener(_this.element);
        _this.options = __assign(__assign({}, MASKITO_DEFAULT_OPTIONS), _this.maskitoOptions);
        _this.preprocessor = maskitoPipe(_this.options.preprocessors);
        _this.postprocessor = maskitoPipe(_this.options.postprocessors);
        _this.teardowns = _this.options.plugins.map(function (plugin) {
          return plugin(_this.element, _this.options);
        });

        _this.ensureValueFitsMask();

        _this.updateHistory(_this.elementState);

        _this.eventListener.listen('keydown', function (event) {
          if (isRedo(event)) {
            event.preventDefault();
            return _this.redo();
          }

          if (isUndo(event)) {
            event.preventDefault();
            return _this.undo();
          }
        });

        if (isBeforeInputEventSupported(element)) {
          _this.eventListener.listen('beforeinput', function (event) {
            var isForward = event.inputType.includes('Forward');

            _this.updateHistory(_this.elementState);

            switch (event.inputType) {
              // historyUndo/historyRedo will not be triggered if value was modified programmatically
              case 'historyUndo':
                event.preventDefault();
                return _this.undo();

              case 'historyRedo':
                event.preventDefault();
                return _this.redo();

              case 'deleteByCut':
              case 'deleteContentBackward':
              case 'deleteContentForward':
                return _this.handleDelete({
                  event: event,
                  isForward: isForward,
                  selection: getNotEmptySelection(_this.elementState, isForward)
                });

              case 'deleteWordForward':
              case 'deleteWordBackward':
                return _this.handleDelete({
                  event: event,
                  isForward: isForward,
                  selection: getWordSelection(_this.elementState, isForward),
                  force: true
                });

              case 'deleteSoftLineBackward':
              case 'deleteSoftLineForward':
              case 'deleteHardLineBackward':
              case 'deleteHardLineForward':
                return _this.handleDelete({
                  event: event,
                  isForward: isForward,
                  selection: getLineSelection(_this.elementState, isForward),
                  force: true
                });

              case 'insertCompositionText':
                return;
              // will be handled inside `compositionend` event

              case 'insertLineBreak':
                return _this.handleEnter(event);

              case 'insertFromPaste':
              case 'insertText':
              case 'insertFromDrop':
              default:
                return _this.handleInsert(event, event.data || '');
            }
          });
        } else {
          /** TODO: drop it after browser support bump (Firefox 87+)
           * Also, replace union types `Event | TypedInputEvent` with `TypedInputEvent` inside:
           *** {@link handleDelete}
           *** {@link handleInsert}
           */
          _this.eventListener.listen('keydown', function (event) {
            return _this.handleKeydown(event);
          });

          _this.eventListener.listen('paste', function (event) {
            var _a;

            return _this.handleInsert(event, ((_a = event.clipboardData) === null || _a === void 0 ? void 0 : _a.getData('text/plain')) || '');
          });
        }

        _this.eventListener.listen('input', function (_a) {
          var inputType = _a.inputType;

          if (inputType === 'insertCompositionText') {
            return; // will be handled inside `compositionend` event
          }

          _this.ensureValueFitsMask();

          _this.updateHistory(_this.elementState);
        });

        _this.eventListener.listen('compositionend', function () {
          _this.ensureValueFitsMask();

          _this.updateHistory(_this.elementState);
        });

        return _this;
      }

      Object.defineProperty(Maskito.prototype, "elementState", {
        get: function () {
          var _a = this.element,
              value = _a.value,
              selectionStart = _a.selectionStart,
              selectionEnd = _a.selectionEnd;
          return {
            value: value,
            selection: [selectionStart || 0, selectionEnd || 0]
          };
        },
        enumerable: false,
        configurable: true
      });
      Object.defineProperty(Maskito.prototype, "maxLength", {
        get: function () {
          var maxLength = this.element.maxLength;
          return maxLength === -1 ? Infinity : maxLength;
        },
        enumerable: false,
        configurable: true
      });

      Maskito.prototype.destroy = function () {
        this.eventListener.destroy();
        this.teardowns.forEach(function (teardown) {
          return teardown === null || teardown === void 0 ? void 0 : teardown();
        });
      };

      Maskito.prototype.updateElementState = function (_a, eventInit) {
        var value = _a.value,
            selection = _a.selection;

        if (eventInit === void 0) {
          eventInit = {
            inputType: 'insertText',
            data: null
          };
        }

        var initialValue = this.elementState.value;
        this.updateValue(value);
        this.updateSelectionRange(selection);

        if (initialValue !== value) {
          this.dispatchInputEvent(eventInit);
        }
      };

      Maskito.prototype.updateSelectionRange = function (_a) {
        var _b, _c;

        var _d = __read(_a, 2),
            from = _d[0],
            to = _d[1];

        if (this.element.selectionStart !== from || this.element.selectionEnd !== to) {
          (_c = (_b = this.element).setSelectionRange) === null || _c === void 0 ? void 0 : _c.call(_b, from, to);
        }
      };

      Maskito.prototype.updateValue = function (value) {
        this.element.value = value;
      };

      Maskito.prototype.ensureValueFitsMask = function () {
        this.updateElementState(maskitoTransform(this.elementState, this.options));
      };

      Maskito.prototype.dispatchInputEvent = function (eventInit) {
        if (eventInit === void 0) {
          eventInit = {
            inputType: 'insertText',
            data: null
          };
        }

        var globalObject = typeof window !== 'undefined' ? window : globalThis; // TODO: replace `globalObject` with `globalThis` after bumping Firefox to 65+
        // @see https://caniuse.com/?search=globalThis

        if (globalObject === null || globalObject === void 0 ? void 0 : globalObject.InputEvent) {
          this.element.dispatchEvent(new InputEvent('input', __assign(__assign({}, eventInit), {
            bubbles: true,
            cancelable: false
          })));
        }
      };

      Maskito.prototype.handleKeydown = function (event) {
        var pressedKey = event.key;
        var isForward = pressedKey === 'Delete';

        switch (pressedKey) {
          case 'Backspace':
          case 'Delete':
            return this.handleDelete({
              event: event,
              isForward: isForward,
              selection: getNotEmptySelection(this.elementState, isForward)
            });

          case 'Enter':
            return this.handleEnter(event);
        }

        if (!isEventProducingCharacter(event)) {
          return;
        }

        this.handleInsert(event, pressedKey);
      };

      Maskito.prototype.handleDelete = function (_a) {
        var event = _a.event,
            selection = _a.selection,
            isForward = _a.isForward,
            _b = _a.force,
            force = _b === void 0 ? false : _b;
        var initialState = {
          value: this.elementState.value,
          selection: selection
        };

        var _c = __read(initialState.selection, 2),
            initialFrom = _c[0],
            initialTo = _c[1];

        var elementState = this.preprocessor({
          elementState: initialState,
          data: ''
        }, isForward ? 'deleteForward' : 'deleteBackward').elementState;
        var maskModel = new MaskModel(elementState, this.options);

        var _d = __read(elementState.selection, 2),
            from = _d[0],
            to = _d[1];

        maskModel.deleteCharacters([from, to]);
        var newElementState = this.postprocessor(maskModel, initialState);
        var newPossibleValue = initialState.value.slice(0, initialFrom) + initialState.value.slice(initialTo);

        if (newPossibleValue === newElementState.value && !force) {
          return;
        }

        event.preventDefault();

        if (areElementValuesEqual(initialState, elementState, maskModel, newElementState)) {
          // User presses Backspace/Delete for the fixed value
          return this.updateSelectionRange(isForward ? [to, to] : [from, from]);
        } // TODO: drop it when `event: Event | TypedInputEvent` => `event: TypedInputEvent`


        var inputTypeFallback = isForward ? 'deleteContentForward' : 'deleteContentBackward';
        this.updateElementState(newElementState, {
          inputType: 'inputType' in event ? event.inputType : inputTypeFallback,
          data: null
        });
        this.updateHistory(newElementState);
      };

      Maskito.prototype.handleInsert = function (event, data) {
        var initialElementState = this.elementState;

        var _a = this.preprocessor({
          data: data,
          elementState: initialElementState
        }, 'insert'),
            elementState = _a.elementState,
            _b = _a.data,
            insertedText = _b === void 0 ? data : _b;

        var maskModel = new MaskModel(elementState, this.options);

        try {
          maskModel.addCharacters(elementState.selection, insertedText);
        } catch (_c) {
          return event.preventDefault();
        }

        var _d = __read(elementState.selection, 2),
            from = _d[0],
            to = _d[1];

        var newPossibleValue = elementState.value.slice(0, from) + data + elementState.value.slice(to);
        var newElementState = this.postprocessor(maskModel, initialElementState);

        if (newElementState.value.length > this.maxLength) {
          return event.preventDefault();
        }

        if (newPossibleValue !== newElementState.value) {
          event.preventDefault();
          this.updateElementState(newElementState, {
            data: data,
            inputType: 'inputType' in event ? event.inputType : 'insertText'
          });
          this.updateHistory(newElementState);
        }
      };

      Maskito.prototype.handleEnter = function (event) {
        if (this.isTextArea) {
          this.handleInsert(event, '\n');
        }
      };

      return Maskito;
    }(MaskHistory);

    exports.MASKITO_DEFAULT_ELEMENT_PREDICATE = MASKITO_DEFAULT_ELEMENT_PREDICATE;
    exports.MASKITO_DEFAULT_OPTIONS = MASKITO_DEFAULT_OPTIONS;
    exports.Maskito = Maskito;
    exports.maskitoPipe = maskitoPipe;
    exports.maskitoTransform = maskitoTransform;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
