"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.JSONFile = void 0;
const jsonc_parser_1 = require("jsonc-parser");
/** @private */
class JSONFile {
    constructor(host, path) {
        this.host = host;
        this.path = path;
        this.content = this.host.readText(this.path);
    }
    get JsonAst() {
        if (this._jsonAst) {
            return this._jsonAst;
        }
        const errors = [];
        this._jsonAst = (0, jsonc_parser_1.parseTree)(this.content, errors, { allowTrailingComma: true });
        if (errors.length) {
            const { error, offset } = errors[0];
            throw new Error(`Failed to parse "${this.path}" as JSON AST Object. ${(0, jsonc_parser_1.printParseErrorCode)(error)} at location: ${offset}.`);
        }
        return this._jsonAst;
    }
    get(jsonPath) {
        const jsonAstNode = this.JsonAst;
        if (!jsonAstNode) {
            return undefined;
        }
        if (jsonPath.length === 0) {
            return (0, jsonc_parser_1.getNodeValue)(jsonAstNode);
        }
        const node = (0, jsonc_parser_1.findNodeAtLocation)(jsonAstNode, jsonPath);
        return node === undefined ? undefined : (0, jsonc_parser_1.getNodeValue)(node);
    }
    modify(jsonPath, value, insertInOrder) {
        let getInsertionIndex;
        if (insertInOrder === undefined) {
            const property = jsonPath.slice(-1)[0];
            getInsertionIndex = (properties) => [...properties, property].sort().findIndex((p) => p === property);
        }
        else if (insertInOrder !== false) {
            getInsertionIndex = insertInOrder;
        }
        const edits = (0, jsonc_parser_1.modify)(this.content, jsonPath, value, {
            getInsertionIndex,
            formattingOptions: {
                insertSpaces: true,
                tabSize: 2,
            },
        });
        this.content = (0, jsonc_parser_1.applyEdits)(this.content, edits);
        this.host.overwrite(this.path, this.content);
        this._jsonAst = undefined;
    }
    remove(jsonPath) {
        if (this.get(jsonPath) !== undefined) {
            this.modify(jsonPath, undefined);
        }
    }
}
exports.JSONFile = JSONFile;
//# sourceMappingURL=data:application/json;base64,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