import { ChangeDetectorRef, Inject, Injectable, } from '@angular/core';
import { Subject } from 'rxjs';
import { takeUntil } from 'rxjs/operators';
import { createChange, createNewChange, noop } from '../util';
import { EventArgumentToken } from './event-argument';
import * as i0 from "@angular/core";
export class IoService {
    constructor(differs, cfr, eventArgument, cdr) {
        this.differs = differs;
        this.cfr = cfr;
        this.eventArgument = eventArgument;
        this.cdr = cdr;
        this.checkInit = this.failInit;
        this.lastComponentInst = null;
        this.inputsDiffer = this.differs.find({}).create();
        this.compFactory = null;
        this.outputsShouldDisconnect$ = new Subject();
        this.outputsChanged = () => false;
    }
    get compRef() {
        return this.compInjector.componentRef;
    }
    get componentInst() {
        return this.compRef ? this.compRef.instance : null;
    }
    get componentInstChanged() {
        if (this.lastComponentInst !== this.componentInst) {
            this.lastComponentInst = this.componentInst;
            return true;
        }
        else {
            return false;
        }
    }
    get compCdr() {
        return this.compRef ? this.compRef.injector.get(ChangeDetectorRef) : null;
    }
    ngOnDestroy() {
        this._disconnectOutputs();
    }
    init(componentInjector, options = {}) {
        this.checkInit = componentInjector ? noop : this.failInit;
        this.compInjector = componentInjector;
        if (options.trackOutputChanges) {
            const outputsDiffer = this.differs.find({}).create();
            this.outputsChanged = (outputs) => !!outputsDiffer.diff(outputs);
        }
    }
    update(inputs, outputs, inputsChanged, outputsChanged) {
        this.checkInit();
        this.updateIO(inputs, outputs);
        const compChanged = this.componentInstChanged;
        if (compChanged || inputsChanged) {
            const inputsChanges = this._getInputsChanges();
            if (inputsChanges) {
                this._updateInputChanges(inputsChanges);
            }
            this.updateInputs(compChanged || !this.lastInputChanges);
        }
        if (compChanged || outputsChanged) {
            this.bindOutputs();
        }
    }
    maybeUpdate() {
        this.checkInit();
        if (this.componentInstChanged) {
            this.updateInputs(true);
            this.bindOutputs();
            return;
        }
        if (this.outputsChanged(this.outputs)) {
            this.bindOutputs();
        }
        if (!this.inputs) {
            return;
        }
        const inputsChanges = this._getInputsChanges();
        if (inputsChanges) {
            const isNotFirstChange = !!this.lastInputChanges;
            this._updateInputChanges(inputsChanges);
            if (isNotFirstChange) {
                this.updateInputs();
            }
        }
    }
    updateIO(inputs, outputs) {
        this.inputs = inputs;
        this.outputs = outputs;
    }
    updateInputs(isFirstChange = false) {
        if (isFirstChange) {
            this._updateCompFactory();
        }
        const compInst = this.componentInst;
        let inputs = this.inputs;
        if (!inputs || !compInst) {
            return;
        }
        inputs = this._resolveInputs(inputs);
        Object.keys(inputs).forEach((p) => (compInst[p] = inputs[p]));
        // Mark component for check to re-render with new inputs
        if (this.compCdr) {
            this.compCdr.markForCheck();
        }
        this.notifyOnInputChanges(this.lastInputChanges, isFirstChange);
    }
    bindOutputs() {
        this._disconnectOutputs();
        const compInst = this.componentInst;
        let outputs = this.outputs;
        if (!outputs || !compInst) {
            return;
        }
        outputs = this._resolveOutputs(outputs);
        Object.keys(outputs)
            .filter((p) => compInst[p])
            .forEach((p) => compInst[p]
            .pipe(takeUntil(this.outputsShouldDisconnect$))
            .subscribe((event) => {
            this.cdr.markForCheck();
            return outputs[p](event);
        }));
    }
    notifyOnInputChanges(changes = {}, forceFirstChanges) {
        // Exit early if component not interested to receive changes
        if (!this.componentInst.ngOnChanges) {
            return;
        }
        if (forceFirstChanges) {
            changes = this._collectFirstChanges();
        }
        this.componentInst.ngOnChanges(changes);
    }
    _disconnectOutputs() {
        this.outputsShouldDisconnect$.next();
    }
    _getInputsChanges() {
        return this.inputsDiffer.diff(this.inputs);
    }
    _updateInputChanges(differ) {
        this.lastInputChanges = this._collectChangesFromDiffer(differ);
    }
    _collectFirstChanges() {
        const changes = {};
        const inputs = this.inputs;
        Object.keys(inputs).forEach((prop) => (changes[prop] = createNewChange(inputs[prop])));
        return this._resolveChanges(changes);
    }
    _collectChangesFromDiffer(differ) {
        const changes = {};
        differ.forEachAddedItem((record) => (changes[record.key] = createNewChange(record.currentValue)));
        differ.forEachChangedItem((record) => (changes[record.key] = createChange(record.currentValue, record.previousValue)));
        return this._resolveChanges(changes);
    }
    _resolveCompFactory() {
        try {
            try {
                return this.cfr.resolveComponentFactory(this.compRef.componentType);
            }
            catch (e) {
                // Fallback if componentType does not exist (happens on NgComponentOutlet)
                return this.cfr.resolveComponentFactory(this.compRef.instance.constructor);
            }
        }
        catch (e) {
            // Factory not available - bailout
            return null;
        }
    }
    _updateCompFactory() {
        this.compFactory = this._resolveCompFactory();
    }
    _resolveInputs(inputs) {
        if (!this.compFactory) {
            return inputs;
        }
        return this._remapIO(inputs, this.compFactory.inputs);
    }
    _resolveOutputs(outputs) {
        outputs = this._processOutputs(outputs);
        if (!this.compFactory) {
            return outputs;
        }
        return this._remapIO(outputs, this.compFactory.outputs);
    }
    _processOutputs(outputs) {
        const processedOutputs = {};
        Object.keys(outputs).forEach((key) => {
            const outputExpr = outputs[key];
            if (typeof outputExpr === 'function') {
                processedOutputs[key] = outputExpr;
            }
            else {
                processedOutputs[key] =
                    outputExpr && this._processOutputArgs(outputExpr);
            }
        });
        return processedOutputs;
    }
    _processOutputArgs(output) {
        const { handler } = output;
        const args = 'args' in output ? output.args || [] : [this.eventArgument];
        return (event) => handler(...args.map((arg) => (arg === this.eventArgument ? event : arg)));
    }
    _resolveChanges(changes) {
        if (!this.compFactory) {
            return changes;
        }
        return this._remapIO(changes, this.compFactory.inputs);
    }
    _remapIO(io, mapping) {
        const newIO = {};
        Object.keys(io).forEach((key) => {
            const newKey = this._findPropByTplInMapping(key, mapping) || key;
            newIO[newKey] = io[key];
        });
        return newIO;
    }
    _findPropByTplInMapping(tplName, mapping) {
        for (const map of mapping) {
            if (map.templateName === tplName) {
                return map.propName;
            }
        }
        return null;
    }
    failInit() {
        throw Error('IoService: ComponentInjector was not set! Please call init() method!');
    }
}
/** @nocollapse */ /** @nocollapse */ IoService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "13.0.2", ngImport: i0, type: IoService, deps: [{ token: i0.KeyValueDiffers }, { token: i0.ComponentFactoryResolver }, { token: EventArgumentToken }, { token: i0.ChangeDetectorRef }], target: i0.ɵɵFactoryTarget.Injectable });
/** @nocollapse */ /** @nocollapse */ IoService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "13.0.2", ngImport: i0, type: IoService });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "13.0.2", ngImport: i0, type: IoService, decorators: [{
            type: Injectable
        }], ctorParameters: function () { return [{ type: i0.KeyValueDiffers }, { type: i0.ComponentFactoryResolver }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [EventArgumentToken]
                }] }, { type: i0.ChangeDetectorRef }]; } });
//# sourceMappingURL=data:application/json;base64,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