export class InputManager {
    constructor(htmlInputElement) {
        this.htmlInputElement = htmlInputElement;
    }
    setCursorAt(position) {
        if (this.htmlInputElement.setSelectionRange) {
            this.htmlInputElement.focus();
            this.htmlInputElement.setSelectionRange(position, position);
        }
        else if (this.htmlInputElement.createTextRange) {
            let textRange = this.htmlInputElement.createTextRange();
            textRange.collapse(true);
            textRange.moveEnd("character", position);
            textRange.moveStart("character", position);
            textRange.select();
        }
    }
    updateValueAndCursor(newRawValue, oldLength, selectionStart) {
        this.rawValue = newRawValue;
        let newLength = newRawValue.length;
        selectionStart = selectionStart - (oldLength - newLength);
        this.setCursorAt(selectionStart);
    }
    get canInputMoreNumbers() {
        let onlyNumbers = this.rawValue.replace(/[^0-9\u0660-\u0669\u06F0-\u06F9]/g, "");
        let haventReachedMaxLength = !(onlyNumbers.length >= this.htmlInputElement.maxLength && this.htmlInputElement.maxLength >= 0);
        let selectionStart = this.inputSelection.selectionStart;
        let selectionEnd = this.inputSelection.selectionEnd;
        let haveNumberSelected = !!(selectionStart != selectionEnd &&
            this.htmlInputElement.value.substring(selectionStart, selectionEnd).match(/[^0-9\u0660-\u0669\u06F0-\u06F9]/));
        let startWithZero = (this.htmlInputElement.value.substring(0, 1) == "0");
        return haventReachedMaxLength || haveNumberSelected || startWithZero;
    }
    get inputSelection() {
        let selectionStart = 0;
        let selectionEnd = 0;
        if (typeof this.htmlInputElement.selectionStart == "number" && typeof this.htmlInputElement.selectionEnd == "number") {
            selectionStart = this.htmlInputElement.selectionStart;
            selectionEnd = this.htmlInputElement.selectionEnd;
        }
        else {
            let range = document.selection.createRange();
            if (range && range.parentElement() == this.htmlInputElement) {
                let lenght = this.htmlInputElement.value.length;
                let normalizedValue = this.htmlInputElement.value.replace(/\r\n/g, "\n");
                let startRange = this.htmlInputElement.createTextRange();
                startRange.moveToBookmark(range.getBookmark());
                let endRange = this.htmlInputElement.createTextRange();
                endRange.collapse(false);
                if (startRange.compareEndPoints("StartToEnd", endRange) > -1) {
                    selectionStart = selectionEnd = lenght;
                }
                else {
                    selectionStart = -startRange.moveStart("character", -lenght);
                    selectionStart += normalizedValue.slice(0, selectionStart).split("\n").length - 1;
                    if (startRange.compareEndPoints("EndToEnd", endRange) > -1) {
                        selectionEnd = lenght;
                    }
                    else {
                        selectionEnd = -startRange.moveEnd("character", -lenght);
                        selectionEnd += normalizedValue.slice(0, selectionEnd).split("\n").length - 1;
                    }
                }
            }
        }
        return {
            selectionStart: selectionStart,
            selectionEnd: selectionEnd
        };
    }
    get rawValue() {
        return this.htmlInputElement && this.htmlInputElement.value;
    }
    set rawValue(value) {
        this._storedRawValue = value;
        if (this.htmlInputElement) {
            this.htmlInputElement.value = value;
        }
    }
    get storedRawValue() {
        return this._storedRawValue || '';
    }
}
//# sourceMappingURL=data:application/json;base64,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