import { InputManager } from "./input.manager";
import { CurrencyMaskInputMode } from "./currency-mask.config";
export class InputService {
    constructor(htmlInputElement, options) {
        this.htmlInputElement = htmlInputElement;
        this.options = options;
        this.SINGLE_DIGIT_REGEX = new RegExp(/^[0-9\u0660-\u0669\u06F0-\u06F9]$/);
        this.ONLY_NUMBERS_REGEX = new RegExp(/[^0-9\u0660-\u0669\u06F0-\u06F9]/g);
        this.PER_AR_NUMBER = new Map();
        this.inputManager = new InputManager(htmlInputElement);
        this.initialize();
    }
    initialize() {
        this.PER_AR_NUMBER.set("\u06F0", "0");
        this.PER_AR_NUMBER.set("\u06F1", "1");
        this.PER_AR_NUMBER.set("\u06F2", "2");
        this.PER_AR_NUMBER.set("\u06F3", "3");
        this.PER_AR_NUMBER.set("\u06F4", "4");
        this.PER_AR_NUMBER.set("\u06F5", "5");
        this.PER_AR_NUMBER.set("\u06F6", "6");
        this.PER_AR_NUMBER.set("\u06F7", "7");
        this.PER_AR_NUMBER.set("\u06F8", "8");
        this.PER_AR_NUMBER.set("\u06F9", "9");
        this.PER_AR_NUMBER.set("\u0660", "0");
        this.PER_AR_NUMBER.set("\u0661", "1");
        this.PER_AR_NUMBER.set("\u0662", "2");
        this.PER_AR_NUMBER.set("\u0663", "3");
        this.PER_AR_NUMBER.set("\u0664", "4");
        this.PER_AR_NUMBER.set("\u0665", "5");
        this.PER_AR_NUMBER.set("\u0666", "6");
        this.PER_AR_NUMBER.set("\u0667", "7");
        this.PER_AR_NUMBER.set("\u0668", "8");
        this.PER_AR_NUMBER.set("\u0669", "9");
    }
    addNumber(keyCode) {
        const { decimal, precision, inputMode } = this.options;
        let keyChar = String.fromCharCode(keyCode);
        const isDecimalChar = keyChar === this.options.decimal;
        if (!this.rawValue) {
            this.rawValue = this.applyMask(false, keyChar);
            let selectionStart = undefined;
            if (inputMode === CurrencyMaskInputMode.NATURAL && precision > 0) {
                selectionStart = this.rawValue.indexOf(decimal);
                if (isDecimalChar) {
                    selectionStart++;
                }
            }
            this.updateFieldValue(selectionStart);
        }
        else {
            let selectionStart = this.inputSelection.selectionStart;
            let selectionEnd = this.inputSelection.selectionEnd;
            const rawValueStart = this.rawValue.substring(0, selectionStart);
            let rawValueEnd = this.rawValue.substring(selectionEnd, this.rawValue.length);
            // In natural mode, replace decimals instead of shifting them.
            const inDecimalPortion = rawValueStart.indexOf(decimal) !== -1;
            if (inputMode === CurrencyMaskInputMode.NATURAL && inDecimalPortion && selectionStart === selectionEnd) {
                rawValueEnd = rawValueEnd.substring(1);
            }
            const newValue = rawValueStart + keyChar + rawValueEnd;
            let nextSelectionStart = selectionStart + 1;
            const isDecimalOrThousands = isDecimalChar || keyChar === this.options.thousands;
            if (isDecimalOrThousands && keyChar === rawValueEnd[0]) {
                // If the cursor is just before the decimal or thousands separator and the user types the
                // decimal or thousands separator, move the cursor past it.
                nextSelectionStart++;
            }
            else if (!this.SINGLE_DIGIT_REGEX.test(keyChar)) {
                // Ignore other non-numbers.
                return;
            }
            this.rawValue = newValue;
            this.updateFieldValue(nextSelectionStart);
        }
    }
    applyMask(isNumber, rawValue, disablePadAndTrim = false) {
        let { allowNegative, decimal, precision, prefix, suffix, thousands, min, max, inputMode } = this.options;
        rawValue = isNumber ? new Number(rawValue).toFixed(precision) : rawValue;
        let onlyNumbers = rawValue.replace(this.ONLY_NUMBERS_REGEX, "");
        if (!onlyNumbers && rawValue !== decimal) {
            return "";
        }
        if (inputMode === CurrencyMaskInputMode.NATURAL && !isNumber && !disablePadAndTrim) {
            rawValue = this.padOrTrimPrecision(rawValue);
            onlyNumbers = rawValue.replace(this.ONLY_NUMBERS_REGEX, "");
        }
        let integerPart = onlyNumbers.slice(0, onlyNumbers.length - precision)
            .replace(/^\u0660*/g, "")
            .replace(/^\u06F0*/g, "")
            .replace(/^0*/g, "");
        if (integerPart == "") {
            integerPart = "0";
        }
        let integerValue = parseInt(integerPart);
        integerPart = integerPart.replace(/\B(?=([0-9\u0660-\u0669\u06F0-\u06F9]{3})+(?![0-9\u0660-\u0669\u06F0-\u06F9]))/g, thousands);
        if (thousands && integerPart.startsWith(thousands)) {
            integerPart = integerPart.substring(1);
        }
        let newRawValue = integerPart;
        let decimalPart = onlyNumbers.slice(onlyNumbers.length - precision);
        let decimalValue = parseInt(decimalPart) || 0;
        let isNegative = rawValue.indexOf("-") > -1;
        // Ensure max is at least as large as min.
        max = (this.isNullOrUndefined(max) || this.isNullOrUndefined(min)) ? max : Math.max(max, min);
        // Ensure precision number works well with more than 2 digits
        // 23 / 100... 233 / 1000 and so on
        const divideBy = Number('1'.padEnd(precision + 1, '0'));
        // Restrict to the min and max values.
        let newValue = integerValue + (decimalValue / divideBy);
        newValue = isNegative ? -newValue : newValue;
        if (!this.isNullOrUndefined(max) && newValue > max) {
            return this.applyMask(true, max + '');
        }
        else if (!this.isNullOrUndefined(min) && newValue < min) {
            return this.applyMask(true, min + '');
        }
        if (precision > 0) {
            if (newRawValue == "0" && decimalPart.length < precision) {
                newRawValue += decimal + "0".repeat(precision - 1) + decimalPart;
            }
            else {
                newRawValue += decimal + decimalPart;
            }
        }
        let isZero = newValue == 0;
        let operator = (isNegative && allowNegative && !isZero) ? "-" : "";
        return operator + prefix + newRawValue + suffix;
    }
    padOrTrimPrecision(rawValue) {
        let { decimal, precision } = this.options;
        let decimalIndex = rawValue.lastIndexOf(decimal);
        if (decimalIndex === -1) {
            decimalIndex = rawValue.length;
            rawValue += decimal;
        }
        let decimalPortion = rawValue.substring(decimalIndex).replace(this.ONLY_NUMBERS_REGEX, "");
        const actualPrecision = decimalPortion.length;
        if (actualPrecision < precision) {
            for (let i = actualPrecision; i < precision; i++) {
                decimalPortion += '0';
            }
        }
        else if (actualPrecision > precision) {
            decimalPortion = decimalPortion.substring(0, decimalPortion.length + precision - actualPrecision);
        }
        return rawValue.substring(0, decimalIndex) + decimal + decimalPortion;
    }
    clearMask(rawValue) {
        if (this.isNullable() && rawValue === "")
            return null;
        let value = (rawValue || "0").replace(this.options.prefix, "").replace(this.options.suffix, "");
        if (this.options.thousands) {
            value = value.replace(new RegExp("\\" + this.options.thousands, "g"), "");
        }
        if (this.options.decimal) {
            value = value.replace(this.options.decimal, ".");
        }
        this.PER_AR_NUMBER.forEach((val, key) => {
            const re = new RegExp(key, "g");
            value = value.replace(re, val);
        });
        return parseFloat(value);
    }
    changeToNegative() {
        if (this.options.allowNegative && this.rawValue != "" && this.rawValue.charAt(0) != "-" && this.value != 0) {
            // Apply the mask to ensure the min and max values are enforced.
            this.rawValue = this.applyMask(false, "-" + this.rawValue);
        }
    }
    changeToPositive() {
        // Apply the mask to ensure the min and max values are enforced.
        this.rawValue = this.applyMask(false, this.rawValue.replace("-", ""));
    }
    removeNumber(keyCode) {
        let { decimal, thousands, prefix, suffix, inputMode } = this.options;
        if (this.isNullable() && this.value == 0) {
            this.rawValue = null;
            return;
        }
        let selectionEnd = this.inputSelection.selectionEnd;
        let selectionStart = this.inputSelection.selectionStart;
        const suffixStart = this.rawValue.length - suffix.length;
        selectionEnd = Math.min(suffixStart, Math.max(selectionEnd, prefix.length));
        selectionStart = Math.min(suffixStart, Math.max(selectionStart, prefix.length));
        // Check if selection was entirely in the prefix or suffix. 
        if (selectionStart === selectionEnd &&
            this.inputSelection.selectionStart !== this.inputSelection.selectionEnd) {
            this.updateFieldValue(selectionStart);
            return;
        }
        let decimalIndex = this.rawValue.indexOf(decimal);
        if (decimalIndex === -1) {
            decimalIndex = this.rawValue.length;
        }
        let shiftSelection = 0;
        let insertChars = '';
        const isCursorInDecimals = decimalIndex < selectionEnd;
        const isCursorImmediatelyAfterDecimalPoint = decimalIndex + 1 === selectionEnd;
        if (selectionEnd === selectionStart) {
            if (keyCode == 8) {
                if (selectionStart <= prefix.length) {
                    return;
                }
                selectionStart--;
                // If previous char isn't a number, go back one more.
                if (!this.rawValue.substr(selectionStart, 1).match(/\d/)) {
                    selectionStart--;
                }
                // In natural mode, jump backwards when in decimal portion of number.
                if (inputMode === CurrencyMaskInputMode.NATURAL && isCursorInDecimals) {
                    shiftSelection = -1;
                    // when removing a single whole number, replace it with 0
                    if (isCursorImmediatelyAfterDecimalPoint && this.value < 10 && this.value > -10) {
                        insertChars += '0';
                    }
                }
            }
            else if (keyCode == 46 || keyCode == 63272) {
                if (selectionStart === suffixStart) {
                    return;
                }
                selectionEnd++;
                // If next char isn't a number, go one more.
                if (!this.rawValue.substr(selectionStart, 1).match(/\d/)) {
                    selectionStart++;
                    selectionEnd++;
                }
            }
        }
        // In natural mode, replace decimals with 0s.
        if (inputMode === CurrencyMaskInputMode.NATURAL && selectionStart > decimalIndex) {
            const replacedDecimalCount = selectionEnd - selectionStart;
            for (let i = 0; i < replacedDecimalCount; i++) {
                insertChars += '0';
            }
        }
        let selectionFromEnd = this.rawValue.length - selectionEnd;
        this.rawValue = this.rawValue.substring(0, selectionStart) + insertChars + this.rawValue.substring(selectionEnd);
        // Remove leading thousand separator from raw value.
        const startChar = this.rawValue.substr(prefix.length, 1);
        if (startChar === thousands) {
            this.rawValue = this.rawValue.substring(0, prefix.length) + this.rawValue.substring(prefix.length + 1);
            selectionFromEnd = Math.min(selectionFromEnd, this.rawValue.length - prefix.length);
        }
        this.updateFieldValue(this.rawValue.length - selectionFromEnd + shiftSelection, true);
    }
    updateFieldValue(selectionStart, disablePadAndTrim = false) {
        let newRawValue = this.applyMask(false, this.rawValue || "", disablePadAndTrim);
        selectionStart = selectionStart == undefined ? this.rawValue.length : selectionStart;
        selectionStart = Math.max(this.options.prefix.length, Math.min(selectionStart, this.rawValue.length - this.options.suffix.length));
        this.inputManager.updateValueAndCursor(newRawValue, this.rawValue.length, selectionStart);
    }
    updateOptions(options) {
        let value = this.value;
        this.options = options;
        this.value = value;
    }
    prefixLength() {
        return this.options.prefix.length;
    }
    suffixLength() {
        return this.options.suffix.length;
    }
    isNullable() {
        return this.options.nullable;
    }
    get canInputMoreNumbers() {
        return this.inputManager.canInputMoreNumbers;
    }
    get inputSelection() {
        return this.inputManager.inputSelection;
    }
    get rawValue() {
        return this.inputManager.rawValue;
    }
    set rawValue(value) {
        this.inputManager.rawValue = value;
    }
    get storedRawValue() {
        return this.inputManager.storedRawValue;
    }
    get value() {
        return this.clearMask(this.rawValue);
    }
    set value(value) {
        this.rawValue = this.applyMask(true, "" + value);
    }
    isNullOrUndefined(value) {
        return value === null || value === undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5wdXQuc2VydmljZS5qcyIsInNvdXJjZVJvb3QiOiJuZzovL25neC1jdXJyZW5jeS8iLCJzb3VyY2VzIjpbInNyYy9pbnB1dC5zZXJ2aWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLE9BQU8sRUFBRSxZQUFZLEVBQUUsTUFBTSxpQkFBaUIsQ0FBQztBQUMvQyxPQUFPLEVBQXNCLHFCQUFxQixFQUFFLE1BQU0sd0JBQXdCLENBQUM7QUFFbkYsTUFBTSxPQUFPLFlBQVk7SUFnQ3JCLFlBQW9CLGdCQUFxQixFQUFVLE9BQTJCO1FBQTFELHFCQUFnQixHQUFoQixnQkFBZ0IsQ0FBSztRQUFVLFlBQU8sR0FBUCxPQUFPLENBQW9CO1FBL0J0RSx1QkFBa0IsR0FBVyxJQUFJLE1BQU0sQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzdFLHVCQUFrQixHQUFXLElBQUksTUFBTSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFFckYsa0JBQWEsR0FBd0IsSUFBSSxHQUFHLEVBQWtCLENBQUM7UUE2QjNELElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUN2RCxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUE7SUFDckIsQ0FBQztJQTdCRCxVQUFVO1FBQ04sSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRXRDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBU0QsU0FBUyxDQUFDLE9BQWU7UUFDckIsTUFBTSxFQUFDLE9BQU8sRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQztRQUNyRCxJQUFJLE9BQU8sR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzNDLE1BQU0sYUFBYSxHQUFHLE9BQU8sS0FBSyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQztRQUV2RCxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNoQixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQy9DLElBQUksY0FBYyxHQUFVLFNBQVMsQ0FBQztZQUN0QyxJQUFJLFNBQVMsS0FBSyxxQkFBcUIsQ0FBQyxPQUFPLElBQUksU0FBUyxHQUFHLENBQUMsRUFBRTtnQkFDOUQsY0FBYyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUNoRCxJQUFJLGFBQWEsRUFBRTtvQkFDZixjQUFjLEVBQUUsQ0FBQztpQkFDcEI7YUFDSjtZQUNELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsQ0FBQztTQUN6QzthQUFNO1lBQ0gsSUFBSSxjQUFjLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxjQUFjLENBQUM7WUFDeEQsSUFBSSxZQUFZLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUM7WUFDcEQsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1lBQ2pFLElBQUksV0FBVyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBRTlFLDhEQUE4RDtZQUM5RCxNQUFNLGdCQUFnQixHQUFHLGFBQWEsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDL0QsSUFBSSxTQUFTLEtBQUsscUJBQXFCLENBQUMsT0FBTyxJQUFJLGdCQUFnQixJQUFJLGNBQWMsS0FBSyxZQUFZLEVBQUU7Z0JBQ3RHLFdBQVcsR0FBRyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ3hDO1lBRUQsTUFBTSxRQUFRLEdBQUcsYUFBYSxHQUFHLE9BQU8sR0FBRyxXQUFXLENBQUM7WUFDdkQsSUFBSSxrQkFBa0IsR0FBRyxjQUFjLEdBQUcsQ0FBQyxDQUFDO1lBQzVDLE1BQU0sb0JBQW9CLEdBQUcsYUFBYSxJQUFJLE9BQU8sS0FBSyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztZQUNqRixJQUFJLG9CQUFvQixJQUFJLE9BQU8sS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ3BELHlGQUF5RjtnQkFDekYsMkRBQTJEO2dCQUMzRCxrQkFBa0IsRUFBRSxDQUFDO2FBQ3hCO2lCQUFNLElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxFQUFFO2dCQUMvQyw0QkFBNEI7Z0JBQzVCLE9BQU87YUFDVjtZQUVELElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1lBQ3pCLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1NBQzdDO0lBQ0wsQ0FBQztJQUVELFNBQVMsQ0FBQyxRQUFpQixFQUFFLFFBQWdCLEVBQUUsaUJBQWlCLEdBQUcsS0FBSztRQUNwRSxJQUFJLEVBQUMsYUFBYSxFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDO1FBRXZHLFFBQVEsR0FBRyxRQUFRLENBQUMsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDO1FBQ3pFLElBQUksV0FBVyxHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWhFLElBQUksQ0FBQyxXQUFXLElBQUksUUFBUSxLQUFLLE9BQU8sRUFBRTtZQUN0QyxPQUFPLEVBQUUsQ0FBQztTQUNiO1FBRUQsSUFBSSxTQUFTLEtBQUsscUJBQXFCLENBQUMsT0FBTyxJQUFJLENBQUMsUUFBUSxJQUFJLENBQUMsaUJBQWlCLEVBQUU7WUFDaEYsUUFBUSxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUM3QyxXQUFXLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxDQUFDLENBQUM7U0FDL0Q7UUFFRCxJQUFJLFdBQVcsR0FBRyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxXQUFXLENBQUMsTUFBTSxHQUFHLFNBQVMsQ0FBQzthQUNqRSxPQUFPLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQzthQUN4QixPQUFPLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQzthQUN4QixPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRXpCLElBQUksV0FBVyxJQUFJLEVBQUUsRUFBRTtZQUNuQixXQUFXLEdBQUcsR0FBRyxDQUFDO1NBQ3JCO1FBQ0QsSUFBSSxZQUFZLEdBQUcsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBRXpDLFdBQVcsR0FBRyxXQUFXLENBQUMsT0FBTyxDQUFDLGlGQUFpRixFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ2hJLElBQUksU0FBUyxJQUFJLFdBQVcsQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDaEQsV0FBVyxHQUFHLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDMUM7UUFFRCxJQUFJLFdBQVcsR0FBRyxXQUFXLENBQUM7UUFDOUIsSUFBSSxXQUFXLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLFNBQVMsQ0FBQyxDQUFDO1FBQ3BFLElBQUksWUFBWSxHQUFHLFFBQVEsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFOUMsSUFBSSxVQUFVLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUU1QywwQ0FBMEM7UUFDMUMsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRTlGLDZEQUE2RDtRQUM3RCxtQ0FBbUM7UUFDbkMsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsU0FBUyxHQUFHLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRXhELHNDQUFzQztRQUN0QyxJQUFJLFFBQVEsR0FBRyxZQUFZLEdBQUcsQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDLENBQUM7UUFFeEQsUUFBUSxHQUFHLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUM3QyxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxJQUFJLFFBQVEsR0FBRyxHQUFHLEVBQUU7WUFDaEQsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxHQUFHLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDekM7YUFBTSxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxJQUFJLFFBQVEsR0FBRyxHQUFHLEVBQUU7WUFDdkQsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxHQUFHLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDekM7UUFFRCxJQUFJLFNBQVMsR0FBRyxDQUFDLEVBQUU7WUFDZixJQUFJLFdBQVcsSUFBSSxHQUFHLElBQUksV0FBVyxDQUFDLE1BQU0sR0FBRyxTQUFTLEVBQUU7Z0JBQ3RELFdBQVcsSUFBSSxPQUFPLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEdBQUcsQ0FBQyxDQUFDLEdBQUcsV0FBVyxDQUFDO2FBQ3BFO2lCQUFNO2dCQUNILFdBQVcsSUFBSSxPQUFPLEdBQUcsV0FBVyxDQUFDO2FBQ3hDO1NBQ0o7UUFFRCxJQUFJLE1BQU0sR0FBRyxRQUFRLElBQUksQ0FBQyxDQUFDO1FBQzNCLElBQUksUUFBUSxHQUFHLENBQUMsVUFBVSxJQUFJLGFBQWEsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNuRSxPQUFPLFFBQVEsR0FBRyxNQUFNLEdBQUcsV0FBVyxHQUFHLE1BQU0sQ0FBQztJQUNwRCxDQUFDO0lBRUQsa0JBQWtCLENBQUMsUUFBZ0I7UUFDL0IsSUFBSSxFQUFDLE9BQU8sRUFBRSxTQUFTLEVBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDO1FBRXhDLElBQUksWUFBWSxHQUFHLFFBQVEsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDakQsSUFBSSxZQUFZLEtBQUssQ0FBQyxDQUFDLEVBQUU7WUFDckIsWUFBWSxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUM7WUFDL0IsUUFBUSxJQUFJLE9BQU8sQ0FBQztTQUN2QjtRQUVELElBQUksY0FBYyxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMzRixNQUFNLGVBQWUsR0FBRyxjQUFjLENBQUMsTUFBTSxDQUFDO1FBQzlDLElBQUksZUFBZSxHQUFHLFNBQVMsRUFBRTtZQUM3QixLQUFLLElBQUksQ0FBQyxHQUFHLGVBQWUsRUFBRSxDQUFDLEdBQUcsU0FBUyxFQUFFLENBQUMsRUFBRSxFQUFFO2dCQUM5QyxjQUFjLElBQUksR0FBRyxDQUFDO2FBQ3pCO1NBQ0o7YUFBTSxJQUFJLGVBQWUsR0FBRyxTQUFTLEVBQUU7WUFDcEMsY0FBYyxHQUFHLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxNQUFNLEdBQUcsU0FBUyxHQUFHLGVBQWUsQ0FBQyxDQUFDO1NBQ3JHO1FBRUQsT0FBTyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxZQUFZLENBQUMsR0FBRyxPQUFPLEdBQUcsY0FBYyxDQUFDO0lBQzFFLENBQUM7SUFFRCxTQUFTLENBQUMsUUFBZ0I7UUFDdEIsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFLElBQUksUUFBUSxLQUFLLEVBQUU7WUFDcEMsT0FBTyxJQUFJLENBQUM7UUFFaEIsSUFBSSxLQUFLLEdBQUcsQ0FBQyxRQUFRLElBQUksR0FBRyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVoRyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFO1lBQ3hCLEtBQUssR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztTQUM3RTtRQUVELElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUU7WUFDdEIsS0FBSyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDcEQ7UUFFRCxJQUFJLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQVcsRUFBRSxHQUFXLEVBQUUsRUFBRTtZQUNwRCxNQUFNLEVBQUUsR0FBRyxJQUFJLE1BQU0sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7WUFDaEMsS0FBSyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ25DLENBQUMsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDN0IsQ0FBQztJQUVELGdCQUFnQjtRQUNaLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLFFBQVEsSUFBSSxFQUFFLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksR0FBRyxJQUFJLElBQUksQ0FBQyxLQUFLLElBQUksQ0FBQyxFQUFFO1lBQ3hHLGdFQUFnRTtZQUNoRSxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDOUQ7SUFDTCxDQUFDO0lBRUQsZ0JBQWdCO1FBQ1osZ0VBQWdFO1FBQ2hFLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUVELFlBQVksQ0FBQyxPQUFlO1FBQ3hCLElBQUksRUFBQyxPQUFPLEVBQUUsU0FBUyxFQUFFLE1BQU0sRUFBRSxNQUFNLEVBQUUsU0FBUyxFQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQztRQUVuRSxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUUsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLENBQUMsRUFBRTtZQUN0QyxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQztZQUNyQixPQUFPO1NBQ1Y7UUFFRCxJQUFJLFlBQVksR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQztRQUNwRCxJQUFJLGNBQWMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQztRQUV4RCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDO1FBQ3pELFlBQVksR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLFlBQVksRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUM1RSxjQUFjLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFFaEYsNERBQTREO1FBQzVELElBQUksY0FBYyxLQUFLLFlBQVk7WUFDL0IsSUFBSSxDQUFDLGNBQWMsQ0FBQyxjQUFjLEtBQUssSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLEVBQUU7WUFDekUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQ3RDLE9BQU87U0FDVjtRQUVELElBQUksWUFBWSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2xELElBQUksWUFBWSxLQUFLLENBQUMsQ0FBQyxFQUFFO1lBQ3JCLFlBQVksR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQztTQUN2QztRQUVELElBQUksY0FBYyxHQUFHLENBQUMsQ0FBQztRQUN2QixJQUFJLFdBQVcsR0FBRyxFQUFFLENBQUM7UUFDckIsTUFBTSxrQkFBa0IsR0FBRyxZQUFZLEdBQUcsWUFBWSxDQUFDO1FBQ3ZELE1BQU0sb0NBQW9DLEdBQUcsWUFBWSxHQUFHLENBQUMsS0FBSyxZQUFZLENBQUM7UUFDL0UsSUFBSSxZQUFZLEtBQUssY0FBYyxFQUFFO1lBQ2pDLElBQUksT0FBTyxJQUFJLENBQUMsRUFBRTtnQkFDZCxJQUFJLGNBQWMsSUFBSSxNQUFNLENBQUMsTUFBTSxFQUFFO29CQUNqQyxPQUFPO2lCQUNWO2dCQUNELGNBQWMsRUFBRSxDQUFDO2dCQUVqQixxREFBcUQ7Z0JBQ3JELElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUN0RCxjQUFjLEVBQUUsQ0FBQztpQkFDcEI7Z0JBRUQscUVBQXFFO2dCQUNyRSxJQUFJLFNBQVMsS0FBSyxxQkFBcUIsQ0FBQyxPQUFPLElBQUksa0JBQWtCLEVBQUU7b0JBQ25FLGNBQWMsR0FBRyxDQUFDLENBQUMsQ0FBQztvQkFDcEIseURBQXlEO29CQUN6RCxJQUFJLG9DQUFvQyxJQUFJLElBQUksQ0FBQyxLQUFLLEdBQUcsRUFBRSxJQUFJLElBQUksQ0FBQyxLQUFLLEdBQUcsQ0FBQyxFQUFFLEVBQUU7d0JBQzdFLFdBQVcsSUFBSSxHQUFHLENBQUM7cUJBQ3RCO2lCQUNKO2FBQ0o7aUJBQU0sSUFBSSxPQUFPLElBQUksRUFBRSxJQUFJLE9BQU8sSUFBSSxLQUFLLEVBQUU7Z0JBQzFDLElBQUksY0FBYyxLQUFLLFdBQVcsRUFBRTtvQkFDaEMsT0FBTztpQkFDVjtnQkFDRCxZQUFZLEVBQUUsQ0FBQztnQkFFZiw0Q0FBNEM7Z0JBQzVDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUN0RCxjQUFjLEVBQUUsQ0FBQztvQkFDakIsWUFBWSxFQUFFLENBQUM7aUJBQ2xCO2FBQ0o7U0FDSjtRQUVELDZDQUE2QztRQUM3QyxJQUFJLFNBQVMsS0FBSyxxQkFBcUIsQ0FBQyxPQUFPLElBQUksY0FBYyxHQUFHLFlBQVksRUFBRTtZQUM5RSxNQUFNLG9CQUFvQixHQUFHLFlBQVksR0FBRyxjQUFjLENBQUM7WUFDM0QsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLG9CQUFvQixFQUFFLENBQUMsRUFBRSxFQUFFO2dCQUMzQyxXQUFXLElBQUksR0FBRyxDQUFDO2FBQ3RCO1NBQ0o7UUFFRCxJQUFJLGdCQUFnQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxHQUFHLFlBQVksQ0FBQztRQUMzRCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxjQUFjLENBQUMsR0FBRyxXQUFXLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFakgsb0RBQW9EO1FBQ3BELE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDekQsSUFBSSxTQUFTLEtBQUssU0FBUyxFQUFFO1lBQ3pCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3ZHLGdCQUFnQixHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3ZGO1FBRUQsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxHQUFHLGdCQUFnQixHQUFHLGNBQWMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUMxRixDQUFDO0lBRUQsZ0JBQWdCLENBQUMsY0FBdUIsRUFBRSxpQkFBaUIsR0FBRyxLQUFLO1FBQy9ELElBQUksV0FBVyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxRQUFRLElBQUksRUFBRSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDaEYsY0FBYyxHQUFHLGNBQWMsSUFBSSxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUM7UUFDckYsY0FBYyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsY0FBYyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDbkksSUFBSSxDQUFDLFlBQVksQ0FBQyxvQkFBb0IsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7SUFDOUYsQ0FBQztJQUVELGFBQWEsQ0FBQyxPQUFZO1FBQ3RCLElBQUksS0FBSyxHQUFXLElBQUksQ0FBQyxLQUFLLENBQUM7UUFDL0IsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFDdkIsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDdkIsQ0FBQztJQUVELFlBQVk7UUFDUixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQztJQUN0QyxDQUFDO0lBRUQsWUFBWTtRQUNSLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO0lBQ3RDLENBQUM7SUFFRCxVQUFVO1FBQ04sT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUNqQyxDQUFDO0lBRUQsSUFBSSxtQkFBbUI7UUFDbkIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDO0lBQ2pELENBQUM7SUFFRCxJQUFJLGNBQWM7UUFDZCxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDO0lBQzVDLENBQUM7SUFFRCxJQUFJLFFBQVE7UUFDUixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO0lBQ3RDLENBQUM7SUFFRCxJQUFJLFFBQVEsQ0FBQyxLQUFhO1FBQ3RCLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztJQUN2QyxDQUFDO0lBRUQsSUFBSSxjQUFjO1FBQ2QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQztJQUM1QyxDQUFDO0lBRUQsSUFBSSxLQUFLO1FBQ0wsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQsSUFBSSxLQUFLLENBQUMsS0FBYTtRQUNuQixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLEVBQUUsR0FBRyxLQUFLLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRU8saUJBQWlCLENBQUMsS0FBVTtRQUNoQyxPQUFPLEtBQUssS0FBSyxJQUFJLElBQUksS0FBSyxLQUFLLFNBQVMsQ0FBQztJQUNqRCxDQUFDO0NBQ0oiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJbnB1dE1hbmFnZXIgfSBmcm9tIFwiLi9pbnB1dC5tYW5hZ2VyXCI7XG5pbXBvcnQgeyBDdXJyZW5jeU1hc2tDb25maWcsIEN1cnJlbmN5TWFza0lucHV0TW9kZSB9IGZyb20gXCIuL2N1cnJlbmN5LW1hc2suY29uZmlnXCI7XG5cbmV4cG9ydCBjbGFzcyBJbnB1dFNlcnZpY2Uge1xuICAgIHByaXZhdGUgU0lOR0xFX0RJR0lUX1JFR0VYOiBSZWdFeHAgPSBuZXcgUmVnRXhwKC9eWzAtOVxcdTA2NjAtXFx1MDY2OVxcdTA2RjAtXFx1MDZGOV0kLyk7XG4gICAgcHJpdmF0ZSBPTkxZX05VTUJFUlNfUkVHRVg6IFJlZ0V4cCA9IG5ldyBSZWdFeHAoL1teMC05XFx1MDY2MC1cXHUwNjY5XFx1MDZGMC1cXHUwNkY5XS9nKTtcblxuICAgIFBFUl9BUl9OVU1CRVI6IE1hcDxzdHJpbmcsIHN0cmluZz4gPSBuZXcgTWFwPHN0cmluZywgc3RyaW5nPigpO1xuXG4gICAgaW5pdGlhbGl6ZSgpIHtcbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLnNldChcIlxcdTA2RjBcIiwgXCIwXCIpO1xuICAgICAgICB0aGlzLlBFUl9BUl9OVU1CRVIuc2V0KFwiXFx1MDZGMVwiLCBcIjFcIik7XG4gICAgICAgIHRoaXMuUEVSX0FSX05VTUJFUi5zZXQoXCJcXHUwNkYyXCIsIFwiMlwiKTtcbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLnNldChcIlxcdTA2RjNcIiwgXCIzXCIpO1xuICAgICAgICB0aGlzLlBFUl9BUl9OVU1CRVIuc2V0KFwiXFx1MDZGNFwiLCBcIjRcIik7XG4gICAgICAgIHRoaXMuUEVSX0FSX05VTUJFUi5zZXQoXCJcXHUwNkY1XCIsIFwiNVwiKTtcbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLnNldChcIlxcdTA2RjZcIiwgXCI2XCIpO1xuICAgICAgICB0aGlzLlBFUl9BUl9OVU1CRVIuc2V0KFwiXFx1MDZGN1wiLCBcIjdcIik7XG4gICAgICAgIHRoaXMuUEVSX0FSX05VTUJFUi5zZXQoXCJcXHUwNkY4XCIsIFwiOFwiKTtcbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLnNldChcIlxcdTA2RjlcIiwgXCI5XCIpO1xuXG4gICAgICAgIHRoaXMuUEVSX0FSX05VTUJFUi5zZXQoXCJcXHUwNjYwXCIsIFwiMFwiKTtcbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLnNldChcIlxcdTA2NjFcIiwgXCIxXCIpO1xuICAgICAgICB0aGlzLlBFUl9BUl9OVU1CRVIuc2V0KFwiXFx1MDY2MlwiLCBcIjJcIik7XG4gICAgICAgIHRoaXMuUEVSX0FSX05VTUJFUi5zZXQoXCJcXHUwNjYzXCIsIFwiM1wiKTtcbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLnNldChcIlxcdTA2NjRcIiwgXCI0XCIpO1xuICAgICAgICB0aGlzLlBFUl9BUl9OVU1CRVIuc2V0KFwiXFx1MDY2NVwiLCBcIjVcIik7XG4gICAgICAgIHRoaXMuUEVSX0FSX05VTUJFUi5zZXQoXCJcXHUwNjY2XCIsIFwiNlwiKTtcbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLnNldChcIlxcdTA2NjdcIiwgXCI3XCIpO1xuICAgICAgICB0aGlzLlBFUl9BUl9OVU1CRVIuc2V0KFwiXFx1MDY2OFwiLCBcIjhcIik7XG4gICAgICAgIHRoaXMuUEVSX0FSX05VTUJFUi5zZXQoXCJcXHUwNjY5XCIsIFwiOVwiKTtcbiAgICB9XG5cbiAgICBpbnB1dE1hbmFnZXI6IElucHV0TWFuYWdlcjtcblxuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgaHRtbElucHV0RWxlbWVudDogYW55LCBwcml2YXRlIG9wdGlvbnM6IEN1cnJlbmN5TWFza0NvbmZpZykge1xuICAgICAgICB0aGlzLmlucHV0TWFuYWdlciA9IG5ldyBJbnB1dE1hbmFnZXIoaHRtbElucHV0RWxlbWVudCk7XG4gICAgICAgIHRoaXMuaW5pdGlhbGl6ZSgpXG4gICAgfVxuXG4gICAgYWRkTnVtYmVyKGtleUNvZGU6IG51bWJlcik6IHZvaWQge1xuICAgICAgICBjb25zdCB7ZGVjaW1hbCwgcHJlY2lzaW9uLCBpbnB1dE1vZGV9ID0gdGhpcy5vcHRpb25zO1xuICAgICAgICBsZXQga2V5Q2hhciA9IFN0cmluZy5mcm9tQ2hhckNvZGUoa2V5Q29kZSk7XG4gICAgICAgIGNvbnN0IGlzRGVjaW1hbENoYXIgPSBrZXlDaGFyID09PSB0aGlzLm9wdGlvbnMuZGVjaW1hbDtcblxuICAgICAgICBpZiAoIXRoaXMucmF3VmFsdWUpIHtcbiAgICAgICAgICAgIHRoaXMucmF3VmFsdWUgPSB0aGlzLmFwcGx5TWFzayhmYWxzZSwga2V5Q2hhcik7XG4gICAgICAgICAgICBsZXQgc2VsZWN0aW9uU3RhcnQ6bnVtYmVyID0gdW5kZWZpbmVkO1xuICAgICAgICAgICAgaWYgKGlucHV0TW9kZSA9PT0gQ3VycmVuY3lNYXNrSW5wdXRNb2RlLk5BVFVSQUwgJiYgcHJlY2lzaW9uID4gMCkge1xuICAgICAgICAgICAgICAgIHNlbGVjdGlvblN0YXJ0ID0gdGhpcy5yYXdWYWx1ZS5pbmRleE9mKGRlY2ltYWwpO1xuICAgICAgICAgICAgICAgIGlmIChpc0RlY2ltYWxDaGFyKSB7XG4gICAgICAgICAgICAgICAgICAgIHNlbGVjdGlvblN0YXJ0Kys7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgdGhpcy51cGRhdGVGaWVsZFZhbHVlKHNlbGVjdGlvblN0YXJ0KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGxldCBzZWxlY3Rpb25TdGFydCA9IHRoaXMuaW5wdXRTZWxlY3Rpb24uc2VsZWN0aW9uU3RhcnQ7XG4gICAgICAgICAgICBsZXQgc2VsZWN0aW9uRW5kID0gdGhpcy5pbnB1dFNlbGVjdGlvbi5zZWxlY3Rpb25FbmQ7XG4gICAgICAgICAgICBjb25zdCByYXdWYWx1ZVN0YXJ0ID0gdGhpcy5yYXdWYWx1ZS5zdWJzdHJpbmcoMCwgc2VsZWN0aW9uU3RhcnQpO1xuICAgICAgICAgICAgbGV0IHJhd1ZhbHVlRW5kID0gdGhpcy5yYXdWYWx1ZS5zdWJzdHJpbmcoc2VsZWN0aW9uRW5kLCB0aGlzLnJhd1ZhbHVlLmxlbmd0aCk7XG5cbiAgICAgICAgICAgIC8vIEluIG5hdHVyYWwgbW9kZSwgcmVwbGFjZSBkZWNpbWFscyBpbnN0ZWFkIG9mIHNoaWZ0aW5nIHRoZW0uXG4gICAgICAgICAgICBjb25zdCBpbkRlY2ltYWxQb3J0aW9uID0gcmF3VmFsdWVTdGFydC5pbmRleE9mKGRlY2ltYWwpICE9PSAtMTtcbiAgICAgICAgICAgIGlmIChpbnB1dE1vZGUgPT09IEN1cnJlbmN5TWFza0lucHV0TW9kZS5OQVRVUkFMICYmIGluRGVjaW1hbFBvcnRpb24gJiYgc2VsZWN0aW9uU3RhcnQgPT09IHNlbGVjdGlvbkVuZCkge1xuICAgICAgICAgICAgICByYXdWYWx1ZUVuZCA9IHJhd1ZhbHVlRW5kLnN1YnN0cmluZygxKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgY29uc3QgbmV3VmFsdWUgPSByYXdWYWx1ZVN0YXJ0ICsga2V5Q2hhciArIHJhd1ZhbHVlRW5kO1xuICAgICAgICAgICAgbGV0IG5leHRTZWxlY3Rpb25TdGFydCA9IHNlbGVjdGlvblN0YXJ0ICsgMTtcbiAgICAgICAgICAgIGNvbnN0IGlzRGVjaW1hbE9yVGhvdXNhbmRzID0gaXNEZWNpbWFsQ2hhciB8fCBrZXlDaGFyID09PSB0aGlzLm9wdGlvbnMudGhvdXNhbmRzO1xuICAgICAgICAgICAgaWYgKGlzRGVjaW1hbE9yVGhvdXNhbmRzICYmIGtleUNoYXIgPT09IHJhd1ZhbHVlRW5kWzBdKSB7XG4gICAgICAgICAgICAgICAgLy8gSWYgdGhlIGN1cnNvciBpcyBqdXN0IGJlZm9yZSB0aGUgZGVjaW1hbCBvciB0aG91c2FuZHMgc2VwYXJhdG9yIGFuZCB0aGUgdXNlciB0eXBlcyB0aGVcbiAgICAgICAgICAgICAgICAvLyBkZWNpbWFsIG9yIHRob3VzYW5kcyBzZXBhcmF0b3IsIG1vdmUgdGhlIGN1cnNvciBwYXN0IGl0LlxuICAgICAgICAgICAgICAgIG5leHRTZWxlY3Rpb25TdGFydCsrO1xuICAgICAgICAgICAgfSBlbHNlIGlmICghdGhpcy5TSU5HTEVfRElHSVRfUkVHRVgudGVzdChrZXlDaGFyKSkge1xuICAgICAgICAgICAgICAgIC8vIElnbm9yZSBvdGhlciBub24tbnVtYmVycy5cbiAgICAgICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIHRoaXMucmF3VmFsdWUgPSBuZXdWYWx1ZTtcbiAgICAgICAgICAgIHRoaXMudXBkYXRlRmllbGRWYWx1ZShuZXh0U2VsZWN0aW9uU3RhcnQpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgYXBwbHlNYXNrKGlzTnVtYmVyOiBib29sZWFuLCByYXdWYWx1ZTogc3RyaW5nLCBkaXNhYmxlUGFkQW5kVHJpbSA9IGZhbHNlKTogc3RyaW5nIHtcbiAgICAgICAgbGV0IHthbGxvd05lZ2F0aXZlLCBkZWNpbWFsLCBwcmVjaXNpb24sIHByZWZpeCwgc3VmZml4LCB0aG91c2FuZHMsIG1pbiwgbWF4LCBpbnB1dE1vZGV9ID0gdGhpcy5vcHRpb25zO1xuXG4gICAgICAgIHJhd1ZhbHVlID0gaXNOdW1iZXIgPyBuZXcgTnVtYmVyKHJhd1ZhbHVlKS50b0ZpeGVkKHByZWNpc2lvbikgOiByYXdWYWx1ZTtcbiAgICAgICAgbGV0IG9ubHlOdW1iZXJzID0gcmF3VmFsdWUucmVwbGFjZSh0aGlzLk9OTFlfTlVNQkVSU19SRUdFWCwgXCJcIik7XG5cbiAgICAgICAgaWYgKCFvbmx5TnVtYmVycyAmJiByYXdWYWx1ZSAhPT0gZGVjaW1hbCkge1xuICAgICAgICAgICAgcmV0dXJuIFwiXCI7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoaW5wdXRNb2RlID09PSBDdXJyZW5jeU1hc2tJbnB1dE1vZGUuTkFUVVJBTCAmJiAhaXNOdW1iZXIgJiYgIWRpc2FibGVQYWRBbmRUcmltKSB7XG4gICAgICAgICAgICByYXdWYWx1ZSA9IHRoaXMucGFkT3JUcmltUHJlY2lzaW9uKHJhd1ZhbHVlKTtcbiAgICAgICAgICAgIG9ubHlOdW1iZXJzID0gcmF3VmFsdWUucmVwbGFjZSh0aGlzLk9OTFlfTlVNQkVSU19SRUdFWCwgXCJcIik7XG4gICAgICAgIH1cblxuICAgICAgICBsZXQgaW50ZWdlclBhcnQgPSBvbmx5TnVtYmVycy5zbGljZSgwLCBvbmx5TnVtYmVycy5sZW5ndGggLSBwcmVjaXNpb24pXG4gICAgICAgICAgICAucmVwbGFjZSgvXlxcdTA2NjAqL2csIFwiXCIpXG4gICAgICAgICAgICAucmVwbGFjZSgvXlxcdTA2RjAqL2csIFwiXCIpXG4gICAgICAgICAgICAucmVwbGFjZSgvXjAqL2csIFwiXCIpO1xuXG4gICAgICAgIGlmIChpbnRlZ2VyUGFydCA9PSBcIlwiKSB7XG4gICAgICAgICAgICBpbnRlZ2VyUGFydCA9IFwiMFwiO1xuICAgICAgICB9XG4gICAgICAgIGxldCBpbnRlZ2VyVmFsdWUgPSBwYXJzZUludChpbnRlZ2VyUGFydCk7XG5cbiAgICAgICAgaW50ZWdlclBhcnQgPSBpbnRlZ2VyUGFydC5yZXBsYWNlKC9cXEIoPz0oWzAtOVxcdTA2NjAtXFx1MDY2OVxcdTA2RjAtXFx1MDZGOV17M30pKyg/IVswLTlcXHUwNjYwLVxcdTA2NjlcXHUwNkYwLVxcdTA2RjldKSkvZywgdGhvdXNhbmRzKTtcbiAgICAgICAgaWYgKHRob3VzYW5kcyAmJiBpbnRlZ2VyUGFydC5zdGFydHNXaXRoKHRob3VzYW5kcykpIHtcbiAgICAgICAgICAgIGludGVnZXJQYXJ0ID0gaW50ZWdlclBhcnQuc3Vic3RyaW5nKDEpO1xuICAgICAgICB9XG5cbiAgICAgICAgbGV0IG5ld1Jhd1ZhbHVlID0gaW50ZWdlclBhcnQ7XG4gICAgICAgIGxldCBkZWNpbWFsUGFydCA9IG9ubHlOdW1iZXJzLnNsaWNlKG9ubHlOdW1iZXJzLmxlbmd0aCAtIHByZWNpc2lvbik7XG4gICAgICAgIGxldCBkZWNpbWFsVmFsdWUgPSBwYXJzZUludChkZWNpbWFsUGFydCkgfHwgMDtcblxuICAgICAgICBsZXQgaXNOZWdhdGl2ZSA9IHJhd1ZhbHVlLmluZGV4T2YoXCItXCIpID4gLTE7XG5cbiAgICAgICAgLy8gRW5zdXJlIG1heCBpcyBhdCBsZWFzdCBhcyBsYXJnZSBhcyBtaW4uXG4gICAgICAgIG1heCA9ICh0aGlzLmlzTnVsbE9yVW5kZWZpbmVkKG1heCkgfHwgdGhpcy5pc051bGxPclVuZGVmaW5lZChtaW4pKSA/IG1heCA6IE1hdGgubWF4KG1heCwgbWluKTtcblxuICAgICAgICAvLyBFbnN1cmUgcHJlY2lzaW9uIG51bWJlciB3b3JrcyB3ZWxsIHdpdGggbW9yZSB0aGFuIDIgZGlnaXRzXG4gICAgICAgIC8vIDIzIC8gMTAwLi4uIDIzMyAvIDEwMDAgYW5kIHNvIG9uXG4gICAgICAgIGNvbnN0IGRpdmlkZUJ5ID0gTnVtYmVyKCcxJy5wYWRFbmQocHJlY2lzaW9uICsgMSwgJzAnKSk7XG4gICAgICAgIFxuICAgICAgICAvLyBSZXN0cmljdCB0byB0aGUgbWluIGFuZCBtYXggdmFsdWVzLlxuICAgICAgICBsZXQgbmV3VmFsdWUgPSBpbnRlZ2VyVmFsdWUgKyAoZGVjaW1hbFZhbHVlIC8gZGl2aWRlQnkpO1xuXG4gICAgICAgIG5ld1ZhbHVlID0gaXNOZWdhdGl2ZSA/IC1uZXdWYWx1ZSA6IG5ld1ZhbHVlO1xuICAgICAgICBpZiAoIXRoaXMuaXNOdWxsT3JVbmRlZmluZWQobWF4KSAmJiBuZXdWYWx1ZSA+IG1heCkge1xuICAgICAgICAgICAgcmV0dXJuIHRoaXMuYXBwbHlNYXNrKHRydWUsIG1heCArICcnKTtcbiAgICAgICAgfSBlbHNlIGlmICghdGhpcy5pc051bGxPclVuZGVmaW5lZChtaW4pICYmIG5ld1ZhbHVlIDwgbWluKSB7XG4gICAgICAgICAgICByZXR1cm4gdGhpcy5hcHBseU1hc2sodHJ1ZSwgbWluICsgJycpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHByZWNpc2lvbiA+IDApIHtcbiAgICAgICAgICAgIGlmIChuZXdSYXdWYWx1ZSA9PSBcIjBcIiAmJiBkZWNpbWFsUGFydC5sZW5ndGggPCBwcmVjaXNpb24pIHtcbiAgICAgICAgICAgICAgICBuZXdSYXdWYWx1ZSArPSBkZWNpbWFsICsgXCIwXCIucmVwZWF0KHByZWNpc2lvbiAtIDEpICsgZGVjaW1hbFBhcnQ7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICAgIG5ld1Jhd1ZhbHVlICs9IGRlY2ltYWwgKyBkZWNpbWFsUGFydDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIGxldCBpc1plcm8gPSBuZXdWYWx1ZSA9PSAwO1xuICAgICAgICBsZXQgb3BlcmF0b3IgPSAoaXNOZWdhdGl2ZSAmJiBhbGxvd05lZ2F0aXZlICYmICFpc1plcm8pID8gXCItXCIgOiBcIlwiO1xuICAgICAgICByZXR1cm4gb3BlcmF0b3IgKyBwcmVmaXggKyBuZXdSYXdWYWx1ZSArIHN1ZmZpeDtcbiAgICB9XG5cbiAgICBwYWRPclRyaW1QcmVjaXNpb24ocmF3VmFsdWU6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgICAgIGxldCB7ZGVjaW1hbCwgcHJlY2lzaW9ufSA9IHRoaXMub3B0aW9ucztcblxuICAgICAgICBsZXQgZGVjaW1hbEluZGV4ID0gcmF3VmFsdWUubGFzdEluZGV4T2YoZGVjaW1hbCk7XG4gICAgICAgIGlmIChkZWNpbWFsSW5kZXggPT09IC0xKSB7XG4gICAgICAgICAgICBkZWNpbWFsSW5kZXggPSByYXdWYWx1ZS5sZW5ndGg7XG4gICAgICAgICAgICByYXdWYWx1ZSArPSBkZWNpbWFsO1xuICAgICAgICB9XG5cbiAgICAgICAgbGV0IGRlY2ltYWxQb3J0aW9uID0gcmF3VmFsdWUuc3Vic3RyaW5nKGRlY2ltYWxJbmRleCkucmVwbGFjZSh0aGlzLk9OTFlfTlVNQkVSU19SRUdFWCwgXCJcIik7XG4gICAgICAgIGNvbnN0IGFjdHVhbFByZWNpc2lvbiA9IGRlY2ltYWxQb3J0aW9uLmxlbmd0aDtcbiAgICAgICAgaWYgKGFjdHVhbFByZWNpc2lvbiA8IHByZWNpc2lvbikge1xuICAgICAgICAgICAgZm9yIChsZXQgaSA9IGFjdHVhbFByZWNpc2lvbjsgaSA8IHByZWNpc2lvbjsgaSsrKSB7XG4gICAgICAgICAgICAgICAgZGVjaW1hbFBvcnRpb24gKz0gJzAnO1xuICAgICAgICAgICAgfVxuICAgICAgICB9IGVsc2UgaWYgKGFjdHVhbFByZWNpc2lvbiA+IHByZWNpc2lvbikge1xuICAgICAgICAgICAgZGVjaW1hbFBvcnRpb24gPSBkZWNpbWFsUG9ydGlvbi5zdWJzdHJpbmcoMCwgZGVjaW1hbFBvcnRpb24ubGVuZ3RoICsgcHJlY2lzaW9uIC0gYWN0dWFsUHJlY2lzaW9uKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiByYXdWYWx1ZS5zdWJzdHJpbmcoMCwgZGVjaW1hbEluZGV4KSArIGRlY2ltYWwgKyBkZWNpbWFsUG9ydGlvbjtcbiAgICB9XG5cbiAgICBjbGVhck1hc2socmF3VmFsdWU6IHN0cmluZyk6IG51bWJlciB7XG4gICAgICAgIGlmICh0aGlzLmlzTnVsbGFibGUoKSAmJiByYXdWYWx1ZSA9PT0gXCJcIilcbiAgICAgICAgICAgIHJldHVybiBudWxsO1xuXG4gICAgICAgIGxldCB2YWx1ZSA9IChyYXdWYWx1ZSB8fCBcIjBcIikucmVwbGFjZSh0aGlzLm9wdGlvbnMucHJlZml4LCBcIlwiKS5yZXBsYWNlKHRoaXMub3B0aW9ucy5zdWZmaXgsIFwiXCIpO1xuXG4gICAgICAgIGlmICh0aGlzLm9wdGlvbnMudGhvdXNhbmRzKSB7XG4gICAgICAgICAgICB2YWx1ZSA9IHZhbHVlLnJlcGxhY2UobmV3IFJlZ0V4cChcIlxcXFxcIiArIHRoaXMub3B0aW9ucy50aG91c2FuZHMsIFwiZ1wiKSwgXCJcIik7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAodGhpcy5vcHRpb25zLmRlY2ltYWwpIHtcbiAgICAgICAgICAgIHZhbHVlID0gdmFsdWUucmVwbGFjZSh0aGlzLm9wdGlvbnMuZGVjaW1hbCwgXCIuXCIpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5QRVJfQVJfTlVNQkVSLmZvckVhY2goKHZhbDogc3RyaW5nLCBrZXk6IHN0cmluZykgPT4ge1xuICAgICAgICAgICAgY29uc3QgcmUgPSBuZXcgUmVnRXhwKGtleSwgXCJnXCIpO1xuICAgICAgICAgICAgdmFsdWUgPSB2YWx1ZS5yZXBsYWNlKHJlLCB2YWwpO1xuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIHBhcnNlRmxvYXQodmFsdWUpO1xuICAgIH1cblxuICAgIGNoYW5nZVRvTmVnYXRpdmUoKTogdm9pZCB7XG4gICAgICAgIGlmICh0aGlzLm9wdGlvbnMuYWxsb3dOZWdhdGl2ZSAmJiB0aGlzLnJhd1ZhbHVlICE9IFwiXCIgJiYgdGhpcy5yYXdWYWx1ZS5jaGFyQXQoMCkgIT0gXCItXCIgJiYgdGhpcy52YWx1ZSAhPSAwKSB7XG4gICAgICAgICAgICAvLyBBcHBseSB0aGUgbWFzayB0byBlbnN1cmUgdGhlIG1pbiBhbmQgbWF4IHZhbHVlcyBhcmUgZW5mb3JjZWQuXG4gICAgICAgICAgICB0aGlzLnJhd1ZhbHVlID0gdGhpcy5hcHBseU1hc2soZmFsc2UsIFwiLVwiICsgdGhpcy5yYXdWYWx1ZSk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICBjaGFuZ2VUb1Bvc2l0aXZlKCk6IHZvaWQge1xuICAgICAgICAvLyBBcHBseSB0aGUgbWFzayB0byBlbnN1cmUgdGhlIG1pbiBhbmQgbWF4IHZhbHVlcyBhcmUgZW5mb3JjZWQuXG4gICAgICAgIHRoaXMucmF3VmFsdWUgPSB0aGlzLmFwcGx5TWFzayhmYWxzZSwgdGhpcy5yYXdWYWx1ZS5yZXBsYWNlKFwiLVwiLCBcIlwiKSk7XG4gICAgfVxuXG4gICAgcmVtb3ZlTnVtYmVyKGtleUNvZGU6IG51bWJlcik6IHZvaWQge1xuICAgICAgICBsZXQge2RlY2ltYWwsIHRob3VzYW5kcywgcHJlZml4LCBzdWZmaXgsIGlucHV0TW9kZX0gPSB0aGlzLm9wdGlvbnM7XG5cbiAgICAgICAgaWYgKHRoaXMuaXNOdWxsYWJsZSgpICYmIHRoaXMudmFsdWUgPT0gMCkge1xuICAgICAgICAgICAgdGhpcy5yYXdWYWx1ZSA9IG51bGw7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cblxuICAgICAgICBsZXQgc2VsZWN0aW9uRW5kID0gdGhpcy5pbnB1dFNlbGVjdGlvbi5zZWxlY3Rpb25FbmQ7XG4gICAgICAgIGxldCBzZWxlY3Rpb25TdGFydCA9IHRoaXMuaW5wdXRTZWxlY3Rpb24uc2VsZWN0aW9uU3RhcnQ7XG5cbiAgICAgICAgY29uc3Qgc3VmZml4U3RhcnQgPSB0aGlzLnJhd1ZhbHVlLmxlbmd0aCAtIHN1ZmZpeC5sZW5ndGg7XG4gICAgICAgIHNlbGVjdGlvbkVuZCA9IE1hdGgubWluKHN1ZmZpeFN0YXJ0LCBNYXRoLm1heChzZWxlY3Rpb25FbmQsIHByZWZpeC5sZW5ndGgpKTtcbiAgICAgICAgc2VsZWN0aW9uU3RhcnQgPSBNYXRoLm1pbihzdWZmaXhTdGFydCwgTWF0aC5tYXgoc2VsZWN0aW9uU3RhcnQsIHByZWZpeC5sZW5ndGgpKTtcblxuICAgICAgICAvLyBDaGVjayBpZiBzZWxlY3Rpb24gd2FzIGVudGlyZWx5IGluIHRoZSBwcmVmaXggb3Igc3VmZml4LiBcbiAgICAgICAgaWYgKHNlbGVjdGlvblN0YXJ0ID09PSBzZWxlY3Rpb25FbmQgJiZcbiAgICAgICAgICAgIHRoaXMuaW5wdXRTZWxlY3Rpb24uc2VsZWN0aW9uU3RhcnQgIT09IHRoaXMuaW5wdXRTZWxlY3Rpb24uc2VsZWN0aW9uRW5kKSB7XG4gICAgICAgICAgICB0aGlzLnVwZGF0ZUZpZWxkVmFsdWUoc2VsZWN0aW9uU3RhcnQpO1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG5cbiAgICAgICAgbGV0IGRlY2ltYWxJbmRleCA9IHRoaXMucmF3VmFsdWUuaW5kZXhPZihkZWNpbWFsKTtcbiAgICAgICAgaWYgKGRlY2ltYWxJbmRleCA9PT0gLTEpIHtcbiAgICAgICAgICAgIGRlY2ltYWxJbmRleCA9IHRoaXMucmF3VmFsdWUubGVuZ3RoO1xuICAgICAgICB9XG5cbiAgICAgICAgbGV0IHNoaWZ0U2VsZWN0aW9uID0gMDtcbiAgICAgICAgbGV0IGluc2VydENoYXJzID0gJyc7XG4gICAgICAgIGNvbnN0IGlzQ3Vyc29ySW5EZWNpbWFscyA9IGRlY2ltYWxJbmRleCA8IHNlbGVjdGlvbkVuZDtcbiAgICAgICAgY29uc3QgaXNDdXJzb3JJbW1lZGlhdGVseUFmdGVyRGVjaW1hbFBvaW50ID0gZGVjaW1hbEluZGV4ICsgMSA9PT0gc2VsZWN0aW9uRW5kO1xuICAgICAgICBpZiAoc2VsZWN0aW9uRW5kID09PSBzZWxlY3Rpb25TdGFydCkge1xuICAgICAgICAgICAgaWYgKGtleUNvZGUgPT0gOCkge1xuICAgICAgICAgICAgICAgIGlmIChzZWxlY3Rpb25TdGFydCA8PSBwcmVmaXgubGVuZ3RoKSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgc2VsZWN0aW9uU3RhcnQtLTtcblxuICAgICAgICAgICAgICAgIC8vIElmIHByZXZpb3VzIGNoYXIgaXNuJ3QgYSBudW1iZXIsIGdvIGJhY2sgb25lIG1vcmUuXG4gICAgICAgICAgICAgICAgaWYgKCF0aGlzLnJhd1ZhbHVlLnN1YnN0cihzZWxlY3Rpb25TdGFydCwgMSkubWF0Y2goL1xcZC8pKSB7XG4gICAgICAgICAgICAgICAgICAgIHNlbGVjdGlvblN0YXJ0LS07XG4gICAgICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAgICAgLy8gSW4gbmF0dXJhbCBtb2RlLCBqdW1wIGJhY2t3YXJkcyB3aGVuIGluIGRlY2ltYWwgcG9ydGlvbiBvZiBudW1iZXIuXG4gICAgICAgICAgICAgICAgaWYgKGlucHV0TW9kZSA9PT0gQ3VycmVuY3lNYXNrSW5wdXRNb2RlLk5BVFVSQUwgJiYgaXNDdXJzb3JJbkRlY2ltYWxzKSB7XG4gICAgICAgICAgICAgICAgICAgIHNoaWZ0U2VsZWN0aW9uID0gLTE7XG4gICAgICAgICAgICAgICAgICAgIC8vIHdoZW4gcmVtb3ZpbmcgYSBzaW5nbGUgd2hvbGUgbnVtYmVyLCByZXBsYWNlIGl0IHdpdGggMFxuICAgICAgICAgICAgICAgICAgICBpZiAoaXNDdXJzb3JJbW1lZGlhdGVseUFmdGVyRGVjaW1hbFBvaW50ICYmIHRoaXMudmFsdWUgPCAxMCAmJiB0aGlzLnZhbHVlID4gLTEwKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICBpbnNlcnRDaGFycyArPSAnMCc7XG4gICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9IGVsc2UgaWYgKGtleUNvZGUgPT0gNDYgfHwga2V5Q29kZSA9PSA2MzI3Mikge1xuICAgICAgICAgICAgICAgIGlmIChzZWxlY3Rpb25TdGFydCA9PT0gc3VmZml4U3RhcnQpIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBzZWxlY3Rpb25FbmQrKztcblxuICAgICAgICAgICAgICAgIC8vIElmIG5leHQgY2hhciBpc24ndCBhIG51bWJlciwgZ28gb25lIG1vcmUuXG4gICAgICAgICAgICAgICAgaWYgKCF0aGlzLnJhd1ZhbHVlLnN1YnN0cihzZWxlY3Rpb25TdGFydCwgMSkubWF0Y2goL1xcZC8pKSB7XG4gICAgICAgICAgICAgICAgICAgIHNlbGVjdGlvblN0YXJ0Kys7XG4gICAgICAgICAgICAgICAgICAgIHNlbGVjdGlvbkVuZCsrO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEluIG5hdHVyYWwgbW9kZSwgcmVwbGFjZSBkZWNpbWFscyB3aXRoIDBzLlxuICAgICAgICBpZiAoaW5wdXRNb2RlID09PSBDdXJyZW5jeU1hc2tJbnB1dE1vZGUuTkFUVVJBTCAmJiBzZWxlY3Rpb25TdGFydCA+IGRlY2ltYWxJbmRleCkge1xuICAgICAgICAgICAgY29uc3QgcmVwbGFjZWREZWNpbWFsQ291bnQgPSBzZWxlY3Rpb25FbmQgLSBzZWxlY3Rpb25TdGFydDtcbiAgICAgICAgICAgIGZvciAobGV0IGkgPSAwOyBpIDwgcmVwbGFjZWREZWNpbWFsQ291bnQ7IGkrKykge1xuICAgICAgICAgICAgICAgIGluc2VydENoYXJzICs9ICcwJztcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIGxldCBzZWxlY3Rpb25Gcm9tRW5kID0gdGhpcy5yYXdWYWx1ZS5sZW5ndGggLSBzZWxlY3Rpb25FbmQ7XG4gICAgICAgIHRoaXMucmF3VmFsdWUgPSB0aGlzLnJhd1ZhbHVlLnN1YnN0cmluZygwLCBzZWxlY3Rpb25TdGFydCkgKyBpbnNlcnRDaGFycyArIHRoaXMucmF3VmFsdWUuc3Vic3RyaW5nKHNlbGVjdGlvbkVuZCk7XG5cbiAgICAgICAgLy8gUmVtb3ZlIGxlYWRpbmcgdGhvdXNhbmQgc2VwYXJhdG9yIGZyb20gcmF3IHZhbHVlLlxuICAgICAgICBjb25zdCBzdGFydENoYXIgPSB0aGlzLnJhd1ZhbHVlLnN1YnN0cihwcmVmaXgubGVuZ3RoLCAxKTtcbiAgICAgICAgaWYgKHN0YXJ0Q2hhciA9PT0gdGhvdXNhbmRzKSB7XG4gICAgICAgICAgICB0aGlzLnJhd1ZhbHVlID0gdGhpcy5yYXdWYWx1ZS5zdWJzdHJpbmcoMCwgcHJlZml4Lmxlbmd0aCkgKyB0aGlzLnJhd1ZhbHVlLnN1YnN0cmluZyhwcmVmaXgubGVuZ3RoICsgMSk7XG4gICAgICAgICAgICBzZWxlY3Rpb25Gcm9tRW5kID0gTWF0aC5taW4oc2VsZWN0aW9uRnJvbUVuZCwgdGhpcy5yYXdWYWx1ZS5sZW5ndGggLSBwcmVmaXgubGVuZ3RoKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMudXBkYXRlRmllbGRWYWx1ZSh0aGlzLnJhd1ZhbHVlLmxlbmd0aCAtIHNlbGVjdGlvbkZyb21FbmQgKyBzaGlmdFNlbGVjdGlvbiwgdHJ1ZSk7XG4gICAgfVxuXG4gICAgdXBkYXRlRmllbGRWYWx1ZShzZWxlY3Rpb25TdGFydD86IG51bWJlciwgZGlzYWJsZVBhZEFuZFRyaW0gPSBmYWxzZSk6IHZvaWQge1xuICAgICAgICBsZXQgbmV3UmF3VmFsdWUgPSB0aGlzLmFwcGx5TWFzayhmYWxzZSwgdGhpcy5yYXdWYWx1ZSB8fCBcIlwiLCBkaXNhYmxlUGFkQW5kVHJpbSk7XG4gICAgICAgIHNlbGVjdGlvblN0YXJ0ID0gc2VsZWN0aW9uU3RhcnQgPT0gdW5kZWZpbmVkID8gdGhpcy5yYXdWYWx1ZS5sZW5ndGggOiBzZWxlY3Rpb25TdGFydDtcbiAgICAgICAgc2VsZWN0aW9uU3RhcnQgPSBNYXRoLm1heCh0aGlzLm9wdGlvbnMucHJlZml4Lmxlbmd0aCwgTWF0aC5taW4oc2VsZWN0aW9uU3RhcnQsIHRoaXMucmF3VmFsdWUubGVuZ3RoIC0gdGhpcy5vcHRpb25zLnN1ZmZpeC5sZW5ndGgpKTtcbiAgICAgICAgdGhpcy5pbnB1dE1hbmFnZXIudXBkYXRlVmFsdWVBbmRDdXJzb3IobmV3UmF3VmFsdWUsIHRoaXMucmF3VmFsdWUubGVuZ3RoLCBzZWxlY3Rpb25TdGFydCk7XG4gICAgfVxuXG4gICAgdXBkYXRlT3B0aW9ucyhvcHRpb25zOiBhbnkpOiB2b2lkIHtcbiAgICAgICAgbGV0IHZhbHVlOiBudW1iZXIgPSB0aGlzLnZhbHVlO1xuICAgICAgICB0aGlzLm9wdGlvbnMgPSBvcHRpb25zO1xuICAgICAgICB0aGlzLnZhbHVlID0gdmFsdWU7XG4gICAgfVxuXG4gICAgcHJlZml4TGVuZ3RoKCk6IGFueSB7XG4gICAgICAgIHJldHVybiB0aGlzLm9wdGlvbnMucHJlZml4Lmxlbmd0aDtcbiAgICB9XG5cbiAgICBzdWZmaXhMZW5ndGgoKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHRoaXMub3B0aW9ucy5zdWZmaXgubGVuZ3RoO1xuICAgIH1cblxuICAgIGlzTnVsbGFibGUoKSB7XG4gICAgICAgIHJldHVybiB0aGlzLm9wdGlvbnMubnVsbGFibGU7XG4gICAgfVxuXG4gICAgZ2V0IGNhbklucHV0TW9yZU51bWJlcnMoKTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiB0aGlzLmlucHV0TWFuYWdlci5jYW5JbnB1dE1vcmVOdW1iZXJzO1xuICAgIH1cblxuICAgIGdldCBpbnB1dFNlbGVjdGlvbigpOiBhbnkge1xuICAgICAgICByZXR1cm4gdGhpcy5pbnB1dE1hbmFnZXIuaW5wdXRTZWxlY3Rpb247XG4gICAgfVxuXG4gICAgZ2V0IHJhd1ZhbHVlKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiB0aGlzLmlucHV0TWFuYWdlci5yYXdWYWx1ZTtcbiAgICB9XG5cbiAgICBzZXQgcmF3VmFsdWUodmFsdWU6IHN0cmluZykge1xuICAgICAgICB0aGlzLmlucHV0TWFuYWdlci5yYXdWYWx1ZSA9IHZhbHVlO1xuICAgIH1cblxuICAgIGdldCBzdG9yZWRSYXdWYWx1ZSgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gdGhpcy5pbnB1dE1hbmFnZXIuc3RvcmVkUmF3VmFsdWU7XG4gICAgfVxuXG4gICAgZ2V0IHZhbHVlKCk6IG51bWJlciB7XG4gICAgICAgIHJldHVybiB0aGlzLmNsZWFyTWFzayh0aGlzLnJhd1ZhbHVlKTtcbiAgICB9XG5cbiAgICBzZXQgdmFsdWUodmFsdWU6IG51bWJlcikge1xuICAgICAgICB0aGlzLnJhd1ZhbHVlID0gdGhpcy5hcHBseU1hc2sodHJ1ZSwgXCJcIiArIHZhbHVlKTtcbiAgICB9XG5cbiAgICBwcml2YXRlIGlzTnVsbE9yVW5kZWZpbmVkKHZhbHVlOiBhbnkpIHtcbiAgICAgICAgcmV0dXJuIHZhbHVlID09PSBudWxsIHx8IHZhbHVlID09PSB1bmRlZmluZWQ7XG4gICAgfVxufVxuIl19