import { InputManager } from "./input.manager";
import { CurrencyMaskInputMode } from "./currency-mask.config";
var InputService = /** @class */ (function () {
    function InputService(htmlInputElement, options) {
        this.htmlInputElement = htmlInputElement;
        this.options = options;
        this.SINGLE_DIGIT_REGEX = new RegExp(/^[0-9\u0660-\u0669\u06F0-\u06F9]$/);
        this.ONLY_NUMBERS_REGEX = new RegExp(/[^0-9\u0660-\u0669\u06F0-\u06F9]/g);
        this.PER_AR_NUMBER = new Map();
        this.inputManager = new InputManager(htmlInputElement);
        this.initialize();
    }
    InputService.prototype.initialize = function () {
        this.PER_AR_NUMBER.set("\u06F0", "0");
        this.PER_AR_NUMBER.set("\u06F1", "1");
        this.PER_AR_NUMBER.set("\u06F2", "2");
        this.PER_AR_NUMBER.set("\u06F3", "3");
        this.PER_AR_NUMBER.set("\u06F4", "4");
        this.PER_AR_NUMBER.set("\u06F5", "5");
        this.PER_AR_NUMBER.set("\u06F6", "6");
        this.PER_AR_NUMBER.set("\u06F7", "7");
        this.PER_AR_NUMBER.set("\u06F8", "8");
        this.PER_AR_NUMBER.set("\u06F9", "9");
        this.PER_AR_NUMBER.set("\u0660", "0");
        this.PER_AR_NUMBER.set("\u0661", "1");
        this.PER_AR_NUMBER.set("\u0662", "2");
        this.PER_AR_NUMBER.set("\u0663", "3");
        this.PER_AR_NUMBER.set("\u0664", "4");
        this.PER_AR_NUMBER.set("\u0665", "5");
        this.PER_AR_NUMBER.set("\u0666", "6");
        this.PER_AR_NUMBER.set("\u0667", "7");
        this.PER_AR_NUMBER.set("\u0668", "8");
        this.PER_AR_NUMBER.set("\u0669", "9");
    };
    InputService.prototype.addNumber = function (keyCode) {
        var _a = this.options, decimal = _a.decimal, precision = _a.precision, inputMode = _a.inputMode;
        var keyChar = String.fromCharCode(keyCode);
        var isDecimalChar = keyChar === this.options.decimal;
        if (!this.rawValue) {
            this.rawValue = this.applyMask(false, keyChar);
            var selectionStart = undefined;
            if (inputMode === CurrencyMaskInputMode.NATURAL && precision > 0) {
                selectionStart = this.rawValue.indexOf(decimal);
                if (isDecimalChar) {
                    selectionStart++;
                }
            }
            this.updateFieldValue(selectionStart);
        }
        else {
            var selectionStart = this.inputSelection.selectionStart;
            var selectionEnd = this.inputSelection.selectionEnd;
            var rawValueStart = this.rawValue.substring(0, selectionStart);
            var rawValueEnd = this.rawValue.substring(selectionEnd, this.rawValue.length);
            // In natural mode, replace decimals instead of shifting them.
            var inDecimalPortion = rawValueStart.indexOf(decimal) !== -1;
            if (inputMode === CurrencyMaskInputMode.NATURAL && inDecimalPortion && selectionStart === selectionEnd) {
                rawValueEnd = rawValueEnd.substring(1);
            }
            var newValue = rawValueStart + keyChar + rawValueEnd;
            var nextSelectionStart = selectionStart + 1;
            var isDecimalOrThousands = isDecimalChar || keyChar === this.options.thousands;
            if (isDecimalOrThousands && keyChar === rawValueEnd[0]) {
                // If the cursor is just before the decimal or thousands separator and the user types the
                // decimal or thousands separator, move the cursor past it.
                nextSelectionStart++;
            }
            else if (!this.SINGLE_DIGIT_REGEX.test(keyChar)) {
                // Ignore other non-numbers.
                return;
            }
            this.rawValue = newValue;
            this.updateFieldValue(nextSelectionStart);
        }
    };
    InputService.prototype.applyMask = function (isNumber, rawValue, disablePadAndTrim) {
        if (disablePadAndTrim === void 0) { disablePadAndTrim = false; }
        var _a = this.options, allowNegative = _a.allowNegative, decimal = _a.decimal, precision = _a.precision, prefix = _a.prefix, suffix = _a.suffix, thousands = _a.thousands, min = _a.min, max = _a.max, inputMode = _a.inputMode;
        rawValue = isNumber ? new Number(rawValue).toFixed(precision) : rawValue;
        var onlyNumbers = rawValue.replace(this.ONLY_NUMBERS_REGEX, "");
        if (!onlyNumbers && rawValue !== decimal) {
            return "";
        }
        if (inputMode === CurrencyMaskInputMode.NATURAL && !isNumber && !disablePadAndTrim) {
            rawValue = this.padOrTrimPrecision(rawValue);
            onlyNumbers = rawValue.replace(this.ONLY_NUMBERS_REGEX, "");
        }
        var integerPart = onlyNumbers.slice(0, onlyNumbers.length - precision)
            .replace(/^\u0660*/g, "")
            .replace(/^\u06F0*/g, "")
            .replace(/^0*/g, "");
        if (integerPart == "") {
            integerPart = "0";
        }
        var integerValue = parseInt(integerPart);
        integerPart = integerPart.replace(/\B(?=([0-9\u0660-\u0669\u06F0-\u06F9]{3})+(?![0-9\u0660-\u0669\u06F0-\u06F9]))/g, thousands);
        if (thousands && integerPart.startsWith(thousands)) {
            integerPart = integerPart.substring(1);
        }
        var newRawValue = integerPart;
        var decimalPart = onlyNumbers.slice(onlyNumbers.length - precision);
        var decimalValue = parseInt(decimalPart) || 0;
        var isNegative = rawValue.indexOf("-") > -1;
        // Ensure max is at least as large as min.
        max = (this.isNullOrUndefined(max) || this.isNullOrUndefined(min)) ? max : Math.max(max, min);
        // Ensure precision number works well with more than 2 digits
        // 23 / 100... 233 / 1000 and so on
        var divideBy = Number('1'.padEnd(precision + 1, '0'));
        // Restrict to the min and max values.
        var newValue = integerValue + (decimalValue / divideBy);
        newValue = isNegative ? -newValue : newValue;
        if (!this.isNullOrUndefined(max) && newValue > max) {
            return this.applyMask(true, max + '');
        }
        else if (!this.isNullOrUndefined(min) && newValue < min) {
            return this.applyMask(true, min + '');
        }
        if (precision > 0) {
            if (newRawValue == "0" && decimalPart.length < precision) {
                newRawValue += decimal + "0".repeat(precision - 1) + decimalPart;
            }
            else {
                newRawValue += decimal + decimalPart;
            }
        }
        var isZero = newValue == 0;
        var operator = (isNegative && allowNegative && !isZero) ? "-" : "";
        return operator + prefix + newRawValue + suffix;
    };
    InputService.prototype.padOrTrimPrecision = function (rawValue) {
        var _a = this.options, decimal = _a.decimal, precision = _a.precision;
        var decimalIndex = rawValue.lastIndexOf(decimal);
        if (decimalIndex === -1) {
            decimalIndex = rawValue.length;
            rawValue += decimal;
        }
        var decimalPortion = rawValue.substring(decimalIndex).replace(this.ONLY_NUMBERS_REGEX, "");
        var actualPrecision = decimalPortion.length;
        if (actualPrecision < precision) {
            for (var i = actualPrecision; i < precision; i++) {
                decimalPortion += '0';
            }
        }
        else if (actualPrecision > precision) {
            decimalPortion = decimalPortion.substring(0, decimalPortion.length + precision - actualPrecision);
        }
        return rawValue.substring(0, decimalIndex) + decimal + decimalPortion;
    };
    InputService.prototype.clearMask = function (rawValue) {
        if (this.isNullable() && rawValue === "")
            return null;
        var value = (rawValue || "0").replace(this.options.prefix, "").replace(this.options.suffix, "");
        if (this.options.thousands) {
            value = value.replace(new RegExp("\\" + this.options.thousands, "g"), "");
        }
        if (this.options.decimal) {
            value = value.replace(this.options.decimal, ".");
        }
        this.PER_AR_NUMBER.forEach(function (val, key) {
            var re = new RegExp(key, "g");
            value = value.replace(re, val);
        });
        return parseFloat(value);
    };
    InputService.prototype.changeToNegative = function () {
        if (this.options.allowNegative && this.rawValue != "" && this.rawValue.charAt(0) != "-" && this.value != 0) {
            // Apply the mask to ensure the min and max values are enforced.
            this.rawValue = this.applyMask(false, "-" + this.rawValue);
        }
    };
    InputService.prototype.changeToPositive = function () {
        // Apply the mask to ensure the min and max values are enforced.
        this.rawValue = this.applyMask(false, this.rawValue.replace("-", ""));
    };
    InputService.prototype.removeNumber = function (keyCode) {
        var _a = this.options, decimal = _a.decimal, thousands = _a.thousands, prefix = _a.prefix, suffix = _a.suffix, inputMode = _a.inputMode;
        if (this.isNullable() && this.value == 0) {
            this.rawValue = null;
            return;
        }
        var selectionEnd = this.inputSelection.selectionEnd;
        var selectionStart = this.inputSelection.selectionStart;
        var suffixStart = this.rawValue.length - suffix.length;
        selectionEnd = Math.min(suffixStart, Math.max(selectionEnd, prefix.length));
        selectionStart = Math.min(suffixStart, Math.max(selectionStart, prefix.length));
        // Check if selection was entirely in the prefix or suffix. 
        if (selectionStart === selectionEnd &&
            this.inputSelection.selectionStart !== this.inputSelection.selectionEnd) {
            this.updateFieldValue(selectionStart);
            return;
        }
        var decimalIndex = this.rawValue.indexOf(decimal);
        if (decimalIndex === -1) {
            decimalIndex = this.rawValue.length;
        }
        var shiftSelection = 0;
        var insertChars = '';
        var isCursorInDecimals = decimalIndex < selectionEnd;
        var isCursorImmediatelyAfterDecimalPoint = decimalIndex + 1 === selectionEnd;
        if (selectionEnd === selectionStart) {
            if (keyCode == 8) {
                if (selectionStart <= prefix.length) {
                    return;
                }
                selectionStart--;
                // If previous char isn't a number, go back one more.
                if (!this.rawValue.substr(selectionStart, 1).match(/\d/)) {
                    selectionStart--;
                }
                // In natural mode, jump backwards when in decimal portion of number.
                if (inputMode === CurrencyMaskInputMode.NATURAL && isCursorInDecimals) {
                    shiftSelection = -1;
                    // when removing a single whole number, replace it with 0
                    if (isCursorImmediatelyAfterDecimalPoint && this.value < 10 && this.value > -10) {
                        insertChars += '0';
                    }
                }
            }
            else if (keyCode == 46 || keyCode == 63272) {
                if (selectionStart === suffixStart) {
                    return;
                }
                selectionEnd++;
                // If next char isn't a number, go one more.
                if (!this.rawValue.substr(selectionStart, 1).match(/\d/)) {
                    selectionStart++;
                    selectionEnd++;
                }
            }
        }
        // In natural mode, replace decimals with 0s.
        if (inputMode === CurrencyMaskInputMode.NATURAL && selectionStart > decimalIndex) {
            var replacedDecimalCount = selectionEnd - selectionStart;
            for (var i = 0; i < replacedDecimalCount; i++) {
                insertChars += '0';
            }
        }
        var selectionFromEnd = this.rawValue.length - selectionEnd;
        this.rawValue = this.rawValue.substring(0, selectionStart) + insertChars + this.rawValue.substring(selectionEnd);
        // Remove leading thousand separator from raw value.
        var startChar = this.rawValue.substr(prefix.length, 1);
        if (startChar === thousands) {
            this.rawValue = this.rawValue.substring(0, prefix.length) + this.rawValue.substring(prefix.length + 1);
            selectionFromEnd = Math.min(selectionFromEnd, this.rawValue.length - prefix.length);
        }
        this.updateFieldValue(this.rawValue.length - selectionFromEnd + shiftSelection, true);
    };
    InputService.prototype.updateFieldValue = function (selectionStart, disablePadAndTrim) {
        if (disablePadAndTrim === void 0) { disablePadAndTrim = false; }
        var newRawValue = this.applyMask(false, this.rawValue || "", disablePadAndTrim);
        selectionStart = selectionStart == undefined ? this.rawValue.length : selectionStart;
        selectionStart = Math.max(this.options.prefix.length, Math.min(selectionStart, this.rawValue.length - this.options.suffix.length));
        this.inputManager.updateValueAndCursor(newRawValue, this.rawValue.length, selectionStart);
    };
    InputService.prototype.updateOptions = function (options) {
        var value = this.value;
        this.options = options;
        this.value = value;
    };
    InputService.prototype.prefixLength = function () {
        return this.options.prefix.length;
    };
    InputService.prototype.suffixLength = function () {
        return this.options.suffix.length;
    };
    InputService.prototype.isNullable = function () {
        return this.options.nullable;
    };
    Object.defineProperty(InputService.prototype, "canInputMoreNumbers", {
        get: function () {
            return this.inputManager.canInputMoreNumbers;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(InputService.prototype, "inputSelection", {
        get: function () {
            return this.inputManager.inputSelection;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(InputService.prototype, "rawValue", {
        get: function () {
            return this.inputManager.rawValue;
        },
        set: function (value) {
            this.inputManager.rawValue = value;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(InputService.prototype, "storedRawValue", {
        get: function () {
            return this.inputManager.storedRawValue;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(InputService.prototype, "value", {
        get: function () {
            return this.clearMask(this.rawValue);
        },
        set: function (value) {
            this.rawValue = this.applyMask(true, "" + value);
        },
        enumerable: false,
        configurable: true
    });
    InputService.prototype.isNullOrUndefined = function (value) {
        return value === null || value === undefined;
    };
    return InputService;
}());
export { InputService };
//# sourceMappingURL=data:application/json;base64,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